/*----------------------------------------------------------------------------
 *
 *  FILE NAME
 *      uart.c
 *
 *  DESCRIPTION
 *      Interruptgesteuert und gepufferte Datenbertragung.
 *
 *  AUTHOR
 *
 *  MODIFICATION HISTORY
 *          Date      Name                Modification
 *----------------------------------------------------------------------------
 */

/* ---- INCLUDES ---------------------------------------------------------- */
#include "uart.h"
#include <msp430x14x.h>
#include <msp430def.h>


#define SET_BIT(byte, bit) ((byte) |=  (bit))
#define CLR_BIT(byte, bit) ((byte) &= ~(bit))



/* receive buffer */
static unsigned char abRxBuf[UART_RX_BUF_LEN];
/* transmit buffer */
static unsigned char abTxBuf[UART_TX_BUF_LEN];

/* receive buffer read pointer */
static unsigned char bRxRd;
/* receive buffer write pointer */
static unsigned char bRxWr;
/* transmit buffer read pointer */
static unsigned char bTxRd;
/* transmit buffer write pointer */
static unsigned char bTxWr;

static unsigned char bStatus;

/* number of available bytes in USART transmit buffer */
static volatile long USART_wTxBytesAvail;
/* number of available bytes in USART receive buffer */
static volatile long USART_wRxBytesAvail;

/* ---- GLOBAL FUNCTIONS -------------------------------------------------- */
 /**************************************************************************
  *
  * Name               :  RS232_Init(void)
  *
  * Description        :  Initialisieren der UART
  *                       Baud Rate 9600, 8bit char, no parity
  *                       Clock source=ACLK
  *
  * Input parameters   :  
  *
  * Output parameters  :  
  *
  **************************************************************************
  */
void RS232_Init(void)
{
/*Clock source=SMCLK @8 MHz */
CLR_BIT(IE1, URXIE0|UTXIE0);

/*UART Select */
SET_BIT(UART0_RS232_SEL,UART0_RS232_TX_PIN);	
SET_BIT(UART0_RS232_SEL,UART0_RS232_RX_PIN);


  /* UART control */
  /* SWRST  = 0 << 0 (reset off/reset)               */
  /* MM     = 0 << 1 (no multiprocessor/multipr.)    */
  /* SYNC   = 0 << 2 (uart mode/spi mode)            */
  /* LISTEN = 0 << 3 (no feedback/feedback)          */
  /* CHAR   = 1 << 4 (7 bit/8 bit)                   */
  /* SP     = 0 << 5 (1 stop/2 stop)                 */
  /* PEV    = 0 << 6 (parity odd/parity even)        */
  /* PENA   = 0 << 7 (no parity/parity)              */
  U0CTL = CHAR;

  /* UART transmit control */
  /* SSEL   = 1 << 4 (clock source UCLKI/ACLK/SMCLK/SMCLK) */
  U0TCTL = SSEL0;

  /* UART receive control */
  /* URXWIE = 0 << 2 (all/start character set interrupt flag)            */
  /* URXEIE = 1 << 3 (erroneous characters do not/do set interrupt flag) */
  U0RCTL = URXEIE;


  /* 9.600bd @8MHz: D = 833.33 U0BR = 0x341, U0MCTL = 10101011 (5 bits) */
  U0BR1 = 0x03; U0BR0 = 0x41; U0MCTL = 0xAB;
 
  /* clear flags */
  bStatus = 0;

  /* reset pointers */
  bRxRd = 0;
  bRxWr = 0;
  bTxRd = 0;
  bTxWr = 0;
  USART_wTxBytesAvail = UART_TX_BUF_LEN;
  USART_wRxBytesAvail = 0;

  /* enable USART0 interrupts */ 
  SET_BIT(IE1, URXIE0|UTXIE0);
 
  /* enable USART0 modules */
  SET_BIT(ME1, URXE0|UTXE0);
}


 /**************************************************************************
  *
  * Name               :  UART0_RX(void)
  *
  * Description        :  Interrupt Behandlung fuer empfangene Daten
  *
  * Input parameters   :  
  *
  * Output parameters  :  
  *
  **************************************************************************
  */
#pragma interrupt_handler uart0_rx_isr:UART0RX_VECTOR
void uart0_rx_isr(void)
{
  /* check if there is any space */
  if( USART_wRxBytesAvail != UART_RX_BUF_LEN )
  {
    /* store the byte */
    abRxBuf[bRxWr] = U0RXBUF;
    /* update write pointer */
    bRxWr = (bRxWr+1) & UART_RX_BUF_MSK; 
    ++USART_wRxBytesAvail;  
  }
}

 /***************************************************************************
  *
  * Name               :  UART0_TX(void)
  *
  * Description        :  Interrupt Behandlung fuer gesendete Daten
  *
  * Input parameters   :  
  *
  * Output parameters  :  
  *
  **************************************************************************/
#pragma interrupt_handler uart0_tx_isr:UART0TX_VECTOR
void uart0_tx_isr(void)
{
  if ( (USART_wTxBytesAvail == UART_TX_BUF_LEN))
  {
    /* stop transmit interrupt */
    /*lint -e(413) NULL pointer is correct register address! */
    CLR_BIT(IE1, UTXIE0);
    /* reset status */
    bStatus = 0;
  }
  else
  {
    /* send next byte */
    U0TXBUF = abTxBuf[bTxRd];
    /* update read pointer */
    bTxRd = (bTxRd+1) & UART_TX_BUF_MSK;
    /* increase available */
    ++USART_wTxBytesAvail;
  }
}

 /***************************************************************************
  *
  * Name               :  UART0_Receive(void)
  *
  * Description        :  lesen der ankommenden Daten
  *
  * Input parameters   :  
  *
  * Output parameters  :  Daten im Buffer.
  *
  **************************************************************************/
unsigned char RS232_Receive(void)
{
  unsigned char value;
  
  /* check if there are bytes in receive buffer */
  if( USART_wRxBytesAvail != 0 )
  {
    /* get the byte */
    value = abRxBuf[bRxRd];
    /* update the read pointer */
    bRxRd = (bRxRd + 1) & UART_RX_BUF_MSK;
    /* decrease available */
    --USART_wRxBytesAvail;
  }
  else
  {
    value = 0xFF;
  }
  return value;
}

 /***************************************************************************
  *
  * Name               :  UART0_Transmit(unsigned char data)
  *
  * Description        :  schreiben der zu sendenden Daten
  *
  * Input parameters   :  1 Byte zu sendende Daten
  *
  * Output parameters  :  
  *
  **************************************************************************/
void RS232_Transmit(unsigned char value)
{
  /* check if there is any space */
  if( USART_wTxBytesAvail != 0 )
  {
    /* store the byte at this position */
    abTxBuf[bTxWr] = value;
    /* update write pointer */
    bTxWr = (bTxWr+1) & UART_TX_BUF_MSK;
    /* decrease available */
    --USART_wTxBytesAvail;

    /* if interrupt not running, restart it */
    if(bStatus == 0)
    {
      /* set status */
      bStatus = 1;
      /* start transmit interrupt always */
      SET_BIT(IE1, 0x80/*UTXIE0*/);
      SET_BIT(IFG1, 0x80/*UTXIFG0*/);
     }
    /*return 1; */
  }
  /* start interrupts again */
  /*return 0; */
}



