// --- [owi_lib.c ] ------------------------------------------------------------
//	     tab = 3
//      basiert auf Maxim Application Note 127 / 187 / an187.zip
// ----------------------------------------------------------------------------

#include <avr/io.h>
#include <avr/interrupt.h>
#include <stdlib.h>
#include <inttypes.h>
#include <util/delay.h>
#include "owi_lib.h"
#include "global.h"
#include "eeprom.h"
#include "rs232.h"

// --- lokale Variablen -------------------------------------------------------

// --- lokale Funktionen ------------------------------------------------------
uint8_t OWReset(void);
void    OWWriteByte(unsigned char byte_value);
void    OWWriteBit(unsigned char bit_value);
uint8_t OWReadByte(void);
uint8_t OWReadBit(void);
uint8_t docrc8( uint8_t number_of_bytes_to_read );

// ----------------------------------------------------------------------------
// liest den ROM-Code des in diesem Moment einzigen (!) angeschlossenen One-Wire-Devices,
// der ROM-Code steht anschlieend im globalen Array ow_buffer[].
// Danach wird die CRC geprft
// Rckgabe: - TRUE wenn CRC und Family-Code stimmt 
uint8_t OW_read_rom_code(void)
{
uint8_t i;

OWReset();
OWWriteByte(READ_ROM);
for (i = 0; i < 8; i++) ow_buffer[i] = OWReadByte();
														// nun steht der ROM-Code in ow_buffer
put_c(13);
for (i = 0; i < 8; i++) 						// ROM-Code ausgeben
	{
	put_s(utoa(ow_buffer[i], msg, 16));
	put_c(' ');
	}

if ((ow_buffer[7] == docrc8(7)) && (ow_buffer[0] == DS18B20))
	{
	put_s("ok ");
	return (TRUE);									// TRUE zurckgeben, wenn CRC identisch
	}
else 
	{
	put_s("crc");									// Fehlermeldung geben, wenn CRC nicht stimmt
	return (FALSE);
	}
}


// ----------------------------------------------------------------------------
// liest das komplette Scratchpad eines Temperatursensors aus,
// die Daten steht anschlieend im Array ow_buffer[],
// es wird noch die CRC geprft
uint8_t OW_read_scratchpad(uint8_t idx)
{
uint8_t i;
EEPROM_read_rom_code(idx);						// den zugehrigen ROM-Code aus dem
														// EEprom nach ow_buffer[] einlesen
OWReset();											// ein Reset senden
OWWriteByte(MATCH_ROM);
for (i = 0; i < 8; i++) OWWriteByte(ow_buffer[i]);
OWWriteByte(READ_SCRATCHPAD);
for (i = 0; i < 9; i++) ow_buffer[i] = OWReadByte();
														// nun steht der ROM-Code in ow_buffer
return (ow_buffer[8] == docrc8(8));
}

// ----------------------------------------------------------------------------
// sendet ein SKIP ROM mit nachfolgendem CONVERT an alle mithrenden Sensoren
// danach muss ca. 1 Sekunde gewartet werden, bis die konvertierten Daten aus
// den einzelnen Sensoren ausgelesen werden knnen.
void OW_convert_all(void)
{
uint8_t i = 80;

OWReset();
OWWriteByte(SKIP_ROM);
OWWriteByte(CONVERT);

// ggf. DQ auf HIGH ziehen
// ONE_WIRE_PORT &= ~(1<<ONE_WIRE_POWER_PIN);

while(i--) _delay_ms(10); 						// 0.8 Sekunden warten

// ggf. DQ wieder freigeben
// ONE_WIRE_PORT |= 1<<ONE_WIRE_POWER_PIN;
// rx_flag = FALSE;								// whrend des Wartens empfangene Kommandos verwerfen
}


// ----------------------------------------------------------------------------
// liefert FALSE, wenn im EEPROM kein Platz mehr fr Sensordaten ist,
// sonst den Index fr den nchsten freien Speicherplatz 
// Die Eeprom-Adresse wird aus dem Index ermittelt durch Multiplikation mit 8 !
uint8_t OW_eeprom_free(void)
{
uint8_t i;
for (i = 1; i < (EEPROM_SIZE / 8); i++)	// die Adressen 0 - 7 sind reserviert
	{
	if (ee_read_byte(i * 8) == 0xFF)
		{
		return(i);									// die erste Stelle zurckgeben, an der
		}												// Byte.0 == 0xFF
	}													// denn bei einem Sensor muss hier der Family-Code stehen
return(0);											// wenn kein freier Platz gefunden
}

// ----------------------------------------------------------------------------
// durchsucht das EEPROM, ob der neu angelernte ROM_CODE im Array ow_buffer[] 
// bereits im EEPROM gespeichert ist,
// liefert TRUE, wenn der Code bereits vorhanden ist !
uint8_t OW_rom_code_exists(void)
{
uint8_t i, j, z, result = FALSE;

for(i = 8; i < (EEPROM_SIZE - 8); i = i+8)
	{
	z = 0;
	for (j = 0; j < 8; j++)
		{
		if (ee_read_byte(i + j) == ow_buffer[j]) z++;
		}
	if (z == 8) 
		{
		result = TRUE;								// wenn alle 8 Bytes identisch waren
		break;										// die Suche abbrechen
		}
	}

return (result);
}

// ----------------------------------------------------------------------------
// liest das EEPROM aus, ermittelt die Anzahl der Sensoren (sensor_cnt)
// liest das Messintervall (messintervall) ein.
//
void OW_read_eeprom(void)
{
uint8_t i;
sensor_cnt = 0;
for (i = 8; i < (EEPROM_SIZE - 8); i += 8)
	{
	if ((ee_read_byte(i) == DS18B20))		// der Family-Code muss 0x28 sein
		{
		sensor_cnt++;								// Anzahl der registrierten Sensoren
		}
	}
	
messintervall  = (ee_read_byte(ADR_MESSINTERVALL) << 8); // high-Byte
messintervall +=  ee_read_byte(ADR_MESSINTERVALL + 1);   // low-Byte
														// wenn == 0xFF, dann default-Wert whlen
if (messintervall == 0xFFFF) messintervall = MESSINTERVALL;

start_modus = ee_read_byte(ADR_AUTOSTART);
}

// ----------------------------------------------------------------------------
// wandelt den Messwert, der aus 2 Byte besteht, in einen String (mit einer Nachkommastelle) um.
// Der Messwert steht rechtsbndig in den globalen Variablen ow_buffer[0] und ow_buffer[1],
// (die Werte entsprechen scratchpad.Byte[0] / scratchpad.Byte[1])
// Das Ergebnis wird ber den globalen String msg[] zurckgegeben.
void DS18B20_convert_temperatur(void)
{
int16_t temp16;
uint8_t i = 0;

ow_buffer[1] = ow_buffer[1] << 4;				// lower nibble im high-Byte nach links schieben
															// upper nibble im low-Byte nach rechts schieben und
															// zum high-Byte addieren.
ow_buffer[1] += (ow_buffer[0] >> 4);			// die Vorkommastellen stehen jetzt in ow_buffer[1]
															// da upper-nibble der Nachkommastellen maskieren,
ow_buffer[0] &= 0x0F; 								// die Nachkommastellen stehen in ow_buffer[0]

temp16  = (int16_t) ow_buffer[1] * 10;
// die Nachkommastelle: temp_lo * 0.625 = temp_lo * 625 / 1000 = temp_lo * 640 / 1024
temp16 += (int16_t)ow_buffer[0] * 640 / 1024;
															// Nun das Dezimaltrennzeichen einschmuggeln:
itoa(temp16, msg, 10);								// die Zahl in einen String umwandeln
while(msg[i]) i++;									// die Position der terminierende /0 finden
msg[i] = msg[i - 1];									// die Nachkommastelle nach rechts schieben
msg[i - 1] = '.';										// statt dessen ein Dezimaltrennzeichen einfgen
msg[++i] = 0;											// Am Ende wieder eine /0 anfgen

}

//-----------------------------------------------------------------------------
// 1-Wire Functions to be implemented for a particular platform
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Reset the 1-Wire bus and return the presence of any device
// Return TRUE  : device present
//        FALSE : no device present
// alle Wartezeiten gem. APP_Note 126, Page 2-3 
// Gesamtdauer: DELAY_G + DELAY_H + DELAY_I + DELAY_J = 1 + 480 + 70 + 410 = 921us
uint8_t OWReset(void)
{
uint8_t r;

_delay_us(DELAY_G);

DQ_LOW;												// DQ auf Low ziehen
_delay_us(DELAY_H);
DQ_INPUT;											// DQ als Eingang schalten
DQ_HIGH;												// auf high schalten (Pullup aktivieren)
_delay_us(DELAY_I);								// warten bis zum Lesen der Antwort

r = DQ_READ;										// Antwort einlesen
if (r) r = FALSE; else r = TRUE;				// die Leitung sollte auf LOW gezogen sein
DQ_OUTPUT;											// DQ wieder als Ausgang schalten
_delay_us(DELAY_J);
	
return(r);
}

//-----------------------------------------------------------------------------
// Send 8 bits of data to the 1-Wire bus
//
void OWWriteByte(unsigned char byte_value)
{
uint8_t i = 8;

while(i--)
	{
	OWWriteBit(byte_value & 0x01);			// das rechts stehende Bit senden
	byte_value = byte_value >> 1;				// um 1 Stelle nach rechts schieben
	}
}

//-----------------------------------------------------------------------------
// Send 1 bit of data to the 1-Wire bus
// DG ist durchgngig als Ausgang geschaltet
// alle Wartezeiten gem. APP_Note 126, Page 2-3 
// Gesamtdauer: DELAY_A + DELAY_B = DELAY_C + DELAY_D = 6 + 64 = 60 + 10 = 70us
void OWWriteBit(uint8_t bit_value)
{
cli();
DQ_LOW;												// DQ auf Low ziehen

if (bit_value)										// Write 1
	{
	_delay_us(DELAY_A);							// 5 us auf Low lassen
	DQ_INPUT;										// DQ durch Pullup auf High ziehen
	_delay_us(DELAY_B);							// 40us warten
	}
	
else													// Write 0
	{
	_delay_us(DELAY_C);							// 45us auf Low lassen
	DQ_INPUT;										// DQ durch Pullup auf High setzen
//	DQ_HIGH;
	_delay_us(DELAY_D);
	}

DQ_OUTPUT;											// DQ wieder als Ausgang schalten
DQ_HIGH;
sei();
}

// ----------------------------------------------------------------------------
uint8_t OWReadByte(void)
{
uint8_t i = 8;
uint8_t data = 0;
while(i--)
	{
	data >>= 1;
	if (OWReadBit()) data |= 0x80;
	}
return(data);
}
	

//-----------------------------------------------------------------------------
// Read 1 bit of data from the 1-Wire bus
// Return 1 : bit read is 1
//        0 : bit read is 0
// alle Wartezeiten gem. APP_Note 126, Page 2-3 
// Gesamtdauer: DELAY_A + DELAY_E + DELAY_F = 6 + 9 + 55 = 70us
uint8_t OWReadBit(void)
{
uint8_t r;
cli();
DQ_LOW;												// DQ auf Low ziehen
_delay_us(DELAY_A);								// 6 us warten
DQ_INPUT;											// als Eingang schalten
DQ_HIGH;

_delay_us(DELAY_E);								// warten und dann messen
r = DQ_READ;

_delay_us(DELAY_F);

DQ_OUTPUT;											// DQ wieder als Ausgang schalten

sei();
return (r);

}

// ----------------------------------------------------------------------------
// liest den ROM_Code von Device idx (ROM-ADresse = idx * 8) aus dem EEProm
// und schreibt ihn in das globale Array ow_buffer[]
// gibt TRUE zurck, wenn die CRC stimmt
uint8_t EEPROM_read_rom_code(uint8_t idx)
{
uint8_t i;
idx *= 8;											// jeweils 8 Byte bilden einen ROM_Code

for ( i = 0; i < 8; i++) ow_buffer[i] = ee_read_byte(idx + i);

return (ow_buffer[7] == docrc8(7));			// wenn die CRC stimmt, TRUE zurckgeben
}

// ----------------------------------------------------------------------------
// schreibt den ROM_Code aus ow_buffer[] ins EEPROM an die Position idx
// die EEPROM-Adresse dazu ist mit vorher mit 8 zu multplizieren
void EEPROM_write_rom_code(uint8_t idx)
{
uint8_t i;
idx *= 8;											// jeweils 8 Byte bilden einen ROM_Code

for ( i = 0; i <  8; i++) ee_write_byte((idx + i), ow_buffer[i]);
}

// ----------------------------------------------------------------------------
// vergleicht den ROM_Code aus ow_buffer[] mit dem im EEPROM an der Stelle idx
// liefert TRUE zurck, wenn ROM_CODE in ow_buffer[] == ROM_CODE im Eeprom
// Der ROM-Code steht an der EEPROM-Adresse idx * 8.
uint8_t EEPROM_compare_rom_code(uint8_t idx)
{
uint8_t i;
uint8_t cmp = TRUE;

idx *= 8;											// jeweils 8 Byte bilden einen ROM_Code

for ( i = 0; i < 8; i++) if(!(ee_read_byte(idx + i) == ow_buffer[i])) cmp = FALSE;
return (cmp);
}

/*      --- diese Variante der CRC-Berechnung kostet zu viel SRAM
// TEST BUILD
static unsigned char PROGMEM dscrc_table[] = {
        0, 94,188,226, 97, 63,221,131,194,156,126, 32,163,253, 31, 65,
      157,195, 33,127,252,162, 64, 30, 95,  1,227,189, 62, 96,130,220,
       35,125,159,193, 66, 28,254,160,225,191, 93,  3,128,222, 60, 98,
      190,224,  2, 92,223,129, 99, 61,124, 34,192,158, 29, 67,161,255,
       70, 24,250,164, 39,121,155,197,132,218, 56,102,229,187, 89,  7,
      219,133,103, 57,186,228,  6, 88, 25, 71,165,251,120, 38,196,154,
      101, 59,217,135,  4, 90,184,230,167,249, 27, 69,198,152,122, 36,
      248,166, 68, 26,153,199, 37,123, 58,100,134,216, 91,  5,231,185,
      140,210, 48,110,237,179, 81, 15, 78, 16,242,172, 47,113,147,205,
       17, 79,173,243,112, 46,204,146,211,141,111, 49,178,236, 14, 80,
      175,241, 19, 77,206,144,114, 44,109, 51,209,143, 12, 82,176,238,
       50,108,142,208, 83, 13,239,177,240,174, 76, 18,145,207, 45,115,
      202,148,118, 40,171,245, 23, 73,  8, 86,180,234,105, 55,213,139,
       87,  9,235,181, 54,104,138,212,149,203, 41,119,244,170, 72, 22,
      233,183, 85, 11,136,214, 52,106, 43,117,151,201, 74, 20,246,168,
      116, 42,200,150, 21, 75,169,247,182,232, 10, 84,215,137,107, 53};

//-----------------------------------------------------------------------------
// Calculate the CRC8 of the byte value provided with the current
// global 'crc8' value.
// Returns current global crc8 value
//
void docrc8(unsigned char value)
{
   // See Application Note 27

   crc8 = pgm_read_byte(&dscrc_table[crc8 ^ value]);

}
*/

// --- [ eof ] ----------------------------------------------------------------