// --- [ owi_main.c ] ----------------------------------------------------------
//       tab = 3
//
//      Temperaturmessung mit 1-Wire-Bus / Sensoren DS18B20 (ROM-SEARCH-free)
//
//      Die ROM-Adressen der beteiligten Sensoren sind im EEPROM angelernt.
//      Via RS232 knnen Kommandos angenommen werden:
//      - 0x10 Start der Messungen 
//      - 0x20 Stopp der Messungen 
//      - 0x30 Anzeigen des ROM_Codes des einzigen angeschlossenen Sensors
//      - 0x31 Anlernen eines Sensors (ROM-Code einlesen und im EEPROM speichern)
//      - 0x40 Alle ROM_Codes im EEPROM lschen
//      - 0x50 Gesamtes EEPROM via RS232 ausgeben
//		  - 0x60 Messreihe nach dem Einschalten automatisch starten
//		  - 0x70 nach dem Starten inaktiv, warten auf Startkommando
//      - 0x8d Einstellung des Messintervalls
//             wobei das BIT.7 (128) der Code fr die Einstellung des Messintervalls ist !
//             Der Wert von Bit.0 bis Bit.6 das Messintervall in [Sekunden * 10].
//             0x8F bedeutet also  15 * 10 Sekunden
//             0xFF bedeutet also 127 * 10 Sekunden = 21 Minuten, 10 Sekunden
//      
// ----------------------------------------------------------------------------

#define EXTERN

#include <avr/io.h>
#include <avr/interrupt.h>
#include <inttypes.h>
#include <stdlib.h>
#include <util/delay.h>
#include <stdio.h>
#include "rs232.h"
#include "owi_lib.h"
#include "global.h"
#include "eeprom.h"

// --- lokale Variablen ------------------------------------------------------

// --- lokale Funktionen ------------------------------------------------------

// ----------------------------------------------------------------------------
int main(void)
{
uint8_t i;

LED_DDR  = 1<<LED1; 								// Busy-LED ein

ONE_WIRE_DDR  = (1<<ONE_WIRE_POWER_PIN | 1<<ONE_WIRE_PORT_PIN);
ONE_WIRE_PORT = (1<<ONE_WIRE_POWER_PIN | 1<<ONE_WIRE_PORT_PIN);

// ATMEGA										// Strom sparen, Timer2/1 und ADC abschalten, nur MEGAx8
PRR = (1<<PRTIM2 | 1<<PRTIM1 | 0<<PRTIM0 | 0<<PRSPI | 0<<PRUSART0 | 1<<PRADC);

// --- die Uhr muss einen Sekundentakt vorgeben
CLKPR = (1<<CLKPCE);								// Clockprescale = 1
CLKPR = 0; 
*/

// Der Timer0 wird im Clear Timer on Compare Match Modus betrieben
// Sobald der TCTN0 den Wert in OCR0A berschreiten, wird der TCNT0
// auf 0 zurckgesetzt und eine ISR ausgefhrt:
// Hier wird die Zeit aktualisiert sowie nach Ablauf des Messintervalls ein Flag gesetzt,
// damit anschlieend die Messungen ausgefhrt werden

// ATMEGA
TCCR0A  = (1<<WGM01);							// CTC - Modus 2
TCCR0B  = (1<<CS02 | 1<< CS00);				// prescale 1024
TIMSK0  = (1<<OCIE0A);							// enable COMPA
OCR0A   = (120 - 1);


// --- die Serielle Schnittstelle aktivieren
RS232_init();
asm volatile ("sei");							// fr Timer und RX erforderlich

ee_print_eeprom();								// einen Speicherauszug des Eeproms drucken

// --- im Eeprom gespeicherte Daten auslesen
OW_read_eeprom();									// der start_modus ist im eeprom gespeichert
if (start_modus) pause = FALSE; else pause = TRUE;// die Messungen anhalten, wenn pause = TRUE

while(1)												// Hauptschleife
	{
	if (rx_flag)									// es wurden Daten via rs232 empfangen
		{
		rx_flag = FALSE;							// das Flag lschen
		if (rx_in == 13) continue;				// ein Return ausblenden	
														// whrend in den nachfolgenden Zeilen ein Echo der Terminaleingabe
														// zurckgeschrieben wird, kommen unter Umstnden noch weiter
														// Zeichen ber die serielle Schnittstelle (wie RETURN) - und berschreiben rx_in.
		rx_received = rx_in;						// Darum wird rx_in umkopiert auf rx_received.
														// 
		put_s("rx: 0x");							// als Rckkoppelung ein Echo der Eingabe ausgeben 						
		put_s(utoa(rx_received, msg, 16)); 
		put_c(' ');
		
		if (rx_received == 0x10)				// Start der Messung
			{
			pause = FALSE;
			}
		
		else if (rx_received == 0x20)			// Messung beenden
			{
			pause = TRUE;							// die Messungen anhalten, die Uhr luft weiter
			LED_PORT &= ~(1<<LED1);				// Busy-LED auf Dauerlicht
			}
			
		else if (rx_received == 0x30)			// Sensor-Rom-Code auslesen, nur den Code ausgeben
			{
			OW_read_rom_code();					// 
			}	
			
		else if (rx_received == 0x31)			// Sensor anlernen, Code speichern
			{
			if (OW_read_rom_code())				// wenn der ROM-Code fehlerfrei ist
				{
				if (!(OW_rom_code_exists()))	// und noch nicht vorhanden ist
					{
					if ((i = OW_eeprom_free()))// und noch Platz im Eeprom ist
						{
						EEPROM_write_rom_code(i);// dann im EEPROM ablegen
						sensor_cnt++;				// Anzahl der Sensoren mitzhlen
						put_c('#');
						put_s(utoa(sensor_cnt, msg, 10)); // und ausgeben 
						}
					}
				}
			}

		else if (rx_received == 0x40)			// ROM-Codes lschen
			{											// alles ab Adr 8 mit 0xFF berschreiben
			for (i = 8; i < EEPROM_SIZE; i++) ee_write_byte(i, 0xFF);
			sensor_cnt = 0;
			}
		
		else if (rx_received == 0x50)			// alle ROM-Codes via RS232 ausgeben
			{
			ee_print_eeprom();					// um die aktuellen Eeprom-Wert anzuzeigen
			}

		else if (rx_received == 0x60)			// nach dem Einschalten direkt mit der Messung beginnen
			{
			ee_write_byte(ADR_AUTOSTART, 255);// das ist der default-Wert
			}
			
		else if (rx_received == 0x70)			// nach dem Einschalten inaktiv, warten auf Startkommando
			{
			ee_write_byte(ADR_AUTOSTART, 0);
			}
				
		else if (rx_received > 0x80)			// Messintervall einstellen	
			{
			messintervall = (uint16_t) (rx_received & 127) * 10; 
			ee_write_byte(ADR_MESSINTERVALL,     (uint8_t) (messintervall >> 8));
			ee_write_byte(ADR_MESSINTERVALL + 1, (uint8_t) (messintervall));
			}
			
		else put_s("err");
		
		//EEARH = 0;								// gibts nicht beim ATTINY/ATMEGA48
		EEARL = 0;									// eepromadresse auf 0 stellen
		put_c(13);									// Zeilenvorschub
		}
		
	if (time_flag)									// wenn das Messintervall abgelaufen ist, dann
		{						
		time_flag = FALSE;
		OW_convert_all();							// die Messungen anstoen, es wird 800ms gewartet,	

		for (i = 0; i < sensor_cnt; i++)		// die Messstellen nacheinander auslesen
			{											// dazu die (Sensornummer + 1) bergeben,
			if (OW_read_scratchpad(i + 1))	// wenn Daten ohne CRC-Fehler eingelesen wurden
				{										// den Index um 1 erhhen, da die Bytes 0..7 keine ROM-Daten enthalten
														// das Ergebnis steht in ow_buffer[],
				DS18B20_convert_temperatur();	// von dort holt sich die Funktion die Messwerte und wandelt sie um,
				put_s(msg);							// den String mit der Temperatur ausgeben
				put_c(';');							// zwischen den Messwerten ';' ausgeben
														// an dieser Stelle kann ein LCD zur Datenausgabe eingebaut werden
				}
				
			else put_c(';');						// bei einem Fehler in der Messung nur ein Semikolon ausgeben
			}

		put_s(utoa(time32, msg, 10));			// Am Ende der Zeile die aktuelle Zeit + RETURN ausgeben
		put_c(13);		
		}
	}
}


// ----------------------------------------------------------------------------
// TIMER0_OVF:  3.686400 MHZ / (1024 * 120) -> 33ms  Zhlt VNC_timeout herunter auf 0
// als Grundlage fr die Zeitzhlung muss ein Takt gewhlt werden, aus dem sich
// der Sekundentakt ableiten lsst: Die ISR wird 30x /Sekunde aufgerufen
ISR(TIMER0_COMPA_vect)
{
static uint8_t tick    = 30; 
static uint16_t messen = 1;					// die erste Messung wird direkt nach dem Start
														// des Timers ausgefhrt
if (!(--tick))										// wenn 1 Sekunde um ist (tick = 30)
	{												
	tick = 30;
	time32++;										// die Zeit im Sekundentakt incrementieren

	if (pause) return;							// wenn pause = TRUE,  dann keine Messungen starten
	else 	LED_PIN = 1<< LED1;					// wenn pause = FALSE, dann LED togglen
	
	if (!(--messen))										
		{
		time_flag++;								// das Flag zum Start der Messungen setzen
		messen = messintervall;
		}
	}
}

// --- [ eof ] ----------------------------------------------------------------
