/*****************************************************************************
Title:     STK500v2 compatible bootloader
Author:    Peter Fleury <pfleury@gmx.ch>   http://jump.to/fleury
Compiler:  avr-gcc 3.4.5 or 4.1 / avr-libc 1.4.3 
Hardware:  All AVRs with bootloader support, tested with ATmega8
License:   GNU General Public License 
          
DESCRIPTION:
    This program allows an AVR with bootloader capabilities to 
    read/write its own Flash/EEprom. To enter Programming mode   
    an input pin is checked. If this pin is pulled low, programming mode  
    is entered. If not, normal execution is done from $0000 
    "reset" vector in Application area.
    Size < 500 words, fits into a 512 word bootloader section 
	when compiled with avr-gcc 4.1
   
USAGE:
    - Set AVR MCU type and clock-frequency (F_CPU) in the Makefile.
    - Set baud rate below (AVRISP only works with 115200 bps)
    - compile/link the bootloader with the supplied Makefile
    - program the "Boot Flash section size" (BOOTSZ fuses),
      for boot-size 512 words:  program BOOTSZ1
    - enable the BOOT Reset Vector (program BOOTRST)
    - Upload the hex file to the AVR using any ISP programmer
    - Program Boot Lock Mode 3 (program BootLock 11 and BootLock 12 lock bits)
    - Reset your AVR while keeping PROG_PIN pulled low
    - Start AVRISP Programmer (AVRStudio/Tools/Program AVR)
    - AVRISP will detect the bootloader
    - Program your application FLASH file and optional EEPROM file using AVRISP
    
Note: 
    Erasing the device without flashing, through AVRISP GUI button "Erase Device" 
    is not implemented, due to AVRStudio limitations.
    Flash is always erased before programming.

	AVRdude:
	Please uncomment #define REMOVE_CMD_SPI_MULTI when using AVRdude.
	Comment #define REMOVE_PROGRAM_LOCK_BIT_SUPPORT to reduce code size
	Read Fuse Bits and Read/Write Lock Bits is not supported

NOTES:
    Based on Atmel Application Note AVR109 - Self-programming
    Based on Atmel Application Note AVR068 - STK500v2 Protocol    
          
LICENSE:
    Copyright (C) 2006 Peter Fleury
				  2007 Stefan Tschiggerl, Robert Schilling

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

ENHANCEMENTS:	Bootloader testet and used for a ATMEGA128
               	with External Memory by Stefan Tschiggerl/Robert Schilling.
				A statusbyte is used now and the Watchdog is used to generate
				a Reset before jumping to Main Application.
				The bootloader ist not left until the Main Application is
				programmed. This is done thrqugh checking Byte 0x00 in Flash
				where the reset vector from Main Application is.
				Into the bootlaoder could also be jumped from the 
				Main Application section. Therfore it is necessary to set
				the statusbyte to the value of the definition MESSAGE_START 
				(0x1B) because this is the START-command from AVR Studio for
				an new message. In the Application section also has to be
				received one complete Message which has to be saved in the
				Buffer receivedMessageAppl. Then it is only necessary to
				generate a Watchdog reset. Note that statusbyte and 
				receivedMessageAppl must have the attribute ".noinit" and
				that this section is defined with the same address in the
				makefile from Bootlaoder and Application Section.
				The receivedMessageAppl ist built up at the following:
					receivedMessageAppl[0]......Sequence Number
					receivedMessageAppl[1]......MSB of Message Length
					receivedMessageAppl[2]......LSB of Message Length
					receivedMessageAppl[3]......Token
					receivedMessageAppl[4-x]....Data Buffer
					receivedMessageAppl[x+1]....checksum

				Note:
					Erasing the device with the Button "Erase Device" in
					AVR Studio is now implemented, but there is only erased
					the first page to guarantee that AVR will stay in
					Bootloader and waits for next Program Flash Command.
					So Flash is not completely erased, because earasing
					the whole device will last too long.

					BOOTLOADER now only fits into a 1024 words Boot section

*****************************************************************************/

#include "stk500boot.h"

/*Statusbyte from Application
 MESSAGE_START --> STK500 command recognized
 0xA0 --> Normal Sequence */
volatile unsigned char statusbyte	__attribute__ ((section (".noinit")));
unsigned char receivedMessageAppl[300] __attribute__ ((section (".noinit")));

void initBootloader(void) __attribute__ ((naked)) __attribute__ ((section (".init0")));
void avrReset(void);

void initBootloader(void)
{
	/* Enable XMEM Interface, No Waitstates */
	MCUCR |= (1 << SRE);	

	if(MCUCSR&(1<<WDRF))
	{

		MCUCSR &= ~(1<<WDRF);
		
		if(statusbyte != MESSAGE_START)
			statusbyte = 0xA0;	// Default value: can be any value betwenn 0x00 and 0xFF
	}							// but not MESSAGE_START commando from AVR Studio (0x1B).
								// Problems also occured when using the value 0x00
								// Now there are no more problems with the value 0xA0
	else						
		statusbyte = 0xA0;	// Default value

	if((statusbyte == 0xA0) && (pgm_read_word(0x0000) != 0xFFFF)) // check statusbyte and Flash in Application Section
	{			
		/* Reset vector in Application Section (0x0000) != 0xFFFF --> Application programmed */		
		asm volatile (	"push r1" "\n\t"  // Jump to Reset vector in Application Section
						"push r1" "\n\t" 
						"ret"     "\n\t" 
						::);
	}
}

void __jumpMain (void) __attribute__ ((naked)) __attribute__ ((section (".init9")));

void __jumpMain(void)
{    
    asm volatile ( ".set __stack, %0" :: "i" (RAMEND) );
    asm volatile ( "clr __zero_reg__" );                       // GCC depends on register r1 set to 0
    asm volatile ( "out %0, __zero_reg__" :: "I" (_SFR_IO_ADDR(SREG)) );  // set SREG to 0
    asm volatile ( "rjmp main");                               // jump to main()
}

int main(void)
{	
	address_t       address = 0;
    address_t       eraseAddress = 0;	
	unsigned char   msgParseState = ST_START;
    unsigned int    i = 0;
    unsigned char   checksum = 0;
    unsigned char   seqNum = 0;
    unsigned int    msgLength = 0;
    unsigned char   msgBuffer[285];
    unsigned char   c, *p;


#ifndef REMOVE_BOOTLOADER_LED
    /* PROG_PIN pulled low, indicate with LED that bootloader is active */
    PROGLED_DDR  |= (1<<PROGLED_PIN);
    PROGLED_PORT |= (1<<PROGLED_PIN);
#endif
    /* set baudrate and enable USART receiver and transmiter without interrupts */
#if COMM_MODE == RS232   
#if UART_BAUDRATE_DOUBLE_SPEED
    UART_STATUS_REG   |=  (1 <<UART_DOUBLE_SPEED);
#endif         
    UART_BAUD_RATE_LOW = UART_BAUD_SELECT(BAUDRATE,F_CPU);
    UART_CONTROL_REG   = (1 << UART_ENABLE_RECEIVER) | (1 << UART_ENABLE_TRANSMITTER);
#endif 
	/* Init FT245 - USB */
#if COMM_MODE == USB_FDXX
	TXE_DDR &= ~(1 << TXE_BIT);
	TXE_PORT |= (1 << TXE_BIT);
	RXF_DDR &= ~(1 << RXF_BIT);
	RXF_PORT |= (1 << RXF_BIT);
#endif


	// if a complete Message was received in Application Section
	// then fill the variables which are used in Bootlaoder
	if(statusbyte == MESSAGE_START)
	{	
		seqNum = receivedMessageAppl[0];
		msgLength = (receivedMessageAppl[1]<<8) | (receivedMessageAppl[2]);

		// Fill Message Buffer for Bootloader with Data from Application Sector
		for(i=0;i<msgLength;i++)
			msgBuffer[i]=receivedMessageAppl[i+4];
		
		checksum = receivedMessageAppl[i];

		i=0;	// Reset to expected value
	}
	
		
    /* main loop */
    for(;;)                             
    {   
        /* Collect received bytes to a complete message */
        
		if(statusbyte == MESSAGE_START)
		{
			msgParseState = ST_PROCESS;
			statusbyte = 0;

		}
		else
			msgParseState = ST_START;
			
	    while (msgParseState != ST_PROCESS)
	    {
			c = recchar();
			
            switch (msgParseState)
            {
				case ST_START:			
					if(c == MESSAGE_START)
					{
						msgParseState = ST_GET_SEQ_NUM;
						checksum = MESSAGE_START;
					}
					break;
						
				case ST_GET_SEQ_NUM:
					
					if ((c == 1) || (c == seqNum))
					{
						seqNum = c;
						msgParseState = ST_MSG_SIZE_1;
						checksum ^= c;
					}
					else
						msgParseState = ST_START;
					break;
						
				case ST_MSG_SIZE_1:			    
					msgLength = c<<8;
					msgParseState = ST_MSG_SIZE_2;
					checksum ^= c;
					break;
						
				case ST_MSG_SIZE_2:			
					msgLength |= c;
					msgParseState = ST_GET_TOKEN;
					checksum ^= c;
					break;
					
				case ST_GET_TOKEN:
					if (c == TOKEN)
					{
						msgParseState = ST_GET_DATA;
						checksum ^= c;
						i = 0;
					}
					else
						msgParseState = ST_START;
					break;
				   
				case ST_GET_DATA:			        
					msgBuffer[i++] = c;
					checksum ^= c;
					if (i == msgLength)
						msgParseState = ST_GET_CHECK;
					break;
							
				case ST_GET_CHECK:				
					if(c == checksum)
						msgParseState = ST_PROCESS;					    
					else
						msgParseState = ST_START;
					break;
				}//switch
		}//while(msgParseState)
		
		
		/* Now process the STK500 commands, see Atmel Appnote AVR068 */
		
	    switch (msgBuffer[0])
	    {
#ifndef REMOVE_CMD_SPI_MULTI
			case CMD_SPI_MULTI:
				{
	                unsigned char answerByte = 0;
                    // only Read Signature Bytes implemented, return dummy value for other instructions
					if (msgBuffer[4]== 0x30)
					{						
						unsigned char signatureIndex = msgBuffer[6];						
					
						if (signatureIndex == 0)
							answerByte = (SIGNATURE_BYTES >>16) & 0x000000FF;
						else if ( signatureIndex == 1 )
							answerByte = (SIGNATURE_BYTES >> 8) & 0x000000FF;
						else
							answerByte = SIGNATURE_BYTES & 0x000000FF;
					}					
    				msgLength = 7;
					msgBuffer[1] = STATUS_CMD_OK;
					msgBuffer[2] = 0;					
					msgBuffer[3] = msgBuffer[4];  // Instruction Byte 1
					msgBuffer[4] = msgBuffer[5];  // Instruction Byte 2
					msgBuffer[5] = answerByte;	                
					msgBuffer[6] = STATUS_CMD_OK;
				}
				break;
#endif
            case CMD_SIGN_ON:
				
		        msgLength = 11;		        
		        msgBuffer[1]  = STATUS_CMD_OK;
		        msgBuffer[2]  = 8;
		        msgBuffer[3]  = 'A';
		        msgBuffer[4]  = 'V';
		        msgBuffer[5]  = 'R';
		        msgBuffer[6]  = 'I';
		        msgBuffer[7]  = 'S';
		        msgBuffer[8]  = 'P';
		        msgBuffer[9]  = '_';
		        msgBuffer[10] = '2';
		        break;
	        
	        case CMD_GET_PARAMETER:
	            {
				
					unsigned char value;
	                
    		        switch(msgBuffer[1])
    		        {
    			    case PARAM_BUILD_NUMBER_LOW:
    				    value = CONFIG_PARAM_BUILD_NUMBER_LOW;
    				    break;				    
    			    case PARAM_BUILD_NUMBER_HIGH:
    				    value = CONFIG_PARAM_BUILD_NUMBER_HIGH;
    				    break;				    
    			    case PARAM_HW_VER:
    				    value = CONFIG_PARAM_HW_VER;
    				    break;				    
    			    case PARAM_SW_MAJOR:
    				    value = CONFIG_PARAM_SW_MAJOR;
    				    break;
    			    case PARAM_SW_MINOR:
    				    value = CONFIG_PARAM_SW_MINOR;
    				    break;				
    				default:
    				    value = 0;
    				    break;
    		        }
		            msgLength = 3;		        
		            msgBuffer[1] = STATUS_CMD_OK;
		            msgBuffer[2] = value;
		        }
	            break;
	            
	        case CMD_SET_PARAMETER:
	        case CMD_ENTER_PROGMODE_ISP:
            case CMD_LEAVE_PROGMODE_ISP:                    
		        msgLength = 2;		        
		        msgBuffer[1] = STATUS_CMD_OK;
                break;
				
            case CMD_READ_SIGNATURE_ISP:
                {
                    unsigned char signatureIndex = msgBuffer[4];
                    unsigned char signature;
					
                    if ( signatureIndex == 0 )
                        signature = (SIGNATURE_BYTES >>16) & 0x000000FF;
                    else if ( signatureIndex == 1 )
                        signature = (SIGNATURE_BYTES >> 8) & 0x000000FF;
                    else
                        signature = SIGNATURE_BYTES & 0x000000FF;
					
	                msgLength = 4;
	                msgBuffer[1] = STATUS_CMD_OK;
	                msgBuffer[2] = signature;
	                msgBuffer[3] = STATUS_CMD_OK;	                
	            }
                break;
				
            case CMD_READ_LOCK_ISP:            
                msgLength = 4;
	            msgBuffer[1] = STATUS_CMD_OK;
	            msgBuffer[2] = boot_lock_fuse_bits_get( GET_LOCK_BITS );
	            msgBuffer[3] = STATUS_CMD_OK;	                                                
                break;
            
            case CMD_READ_FUSE_ISP:
                {                    
                    unsigned char fuseBits;                    
                    
                    if (msgBuffer[2] == 0x50)
                    {
                        if ( msgBuffer[3] == 0x08 )
                            fuseBits = boot_lock_fuse_bits_get( GET_EXTENDED_FUSE_BITS );                            
                        else
                            fuseBits = boot_lock_fuse_bits_get( GET_LOW_FUSE_BITS );                            
                    }
                    else 
                        fuseBits = boot_lock_fuse_bits_get( GET_HIGH_FUSE_BITS );
                  
                    msgLength = 4;    
	                msgBuffer[1] = STATUS_CMD_OK;
	                msgBuffer[2] = fuseBits;	                
	                msgBuffer[3] = STATUS_CMD_OK;	                                    
                }
                break;
                
#ifndef REMOVE_PROGRAM_LOCK_BIT_SUPPORT
            case CMD_PROGRAM_LOCK_ISP:
                {
                    unsigned char lockBits = msgBuffer[4];
                    
                    lockBits = (~lockBits) & 0x3C;  // mask BLBxx bits
				    boot_lock_bits_set(lockBits);	// and program it
				    boot_spm_busy_wait();
                    msgLength = 3;
	                msgBuffer[1] = STATUS_CMD_OK;	                
	                msgBuffer[2] = STATUS_CMD_OK;	                                                        
                }
                break;
#endif
            case CMD_CHIP_ERASE_ISP:			
				eraseAddress = 0;
				boot_page_erase(eraseAddress);	// Erase the first page so that AVR will stay
				boot_spm_busy_wait();			// in Bootlaoder until Flash is programmed again
				boot_rww_enable();	
                
	            msgLength = 2;
	            msgBuffer[1] = STATUS_CMD_OK;
                break;
				
            case CMD_LOAD_ADDRESS:
#if defined(RAMPZ)
                address = ( ((address_t)(msgBuffer[1])<<24)|((address_t)(msgBuffer[2])<<16)|((address_t)(msgBuffer[3])<<8)|(msgBuffer[4]) )<<1;
#else
		        address = ( ((msgBuffer[3])<<8)|(msgBuffer[4]) )<<1;  //convert word to byte address
#endif
		        msgLength = 2;
		        msgBuffer[1] = STATUS_CMD_OK;
                break;
                
            case CMD_PROGRAM_FLASH_ISP:
            case CMD_PROGRAM_EEPROM_ISP:                
                {
                    unsigned int  size = ((msgBuffer[1])<<8) | msgBuffer[2];
                    unsigned char *p = msgBuffer+10;
                    unsigned int  data;
                    unsigned char highByte, lowByte;                    
                    address_t     tempaddress = address;
                   
                    if ( msgBuffer[0] == CMD_PROGRAM_FLASH_ISP )
                    {
                		if  (eraseAddress < APP_END) // erase only main section (bootloader protection)
                		{
                			boot_page_erase(eraseAddress);	// Perform page erase
                			boot_spm_busy_wait();		// Wait until the memory is erased.
                			eraseAddress += SPM_PAGESIZE;    // point to next page to be erase
                		}
                        /* Write FLASH */
        		        do 
						{
        		            lowByte   = *p++;
        		            highByte  = *p++;
        				    data =  (highByte << 8) | lowByte;
        				    boot_page_fill(address,data);
        					address = address + 2;  	// Select next word in memory
        				    size -= 2;			// Reduce number of bytes to write by two    
        			    } while(size);			// Loop until all bytes written
						
        			    boot_page_write(tempaddress);
        	    		boot_spm_busy_wait();	
        		    	boot_rww_enable();				// Re-enable the RWW section                    
    		        }
    		        else
    		           eeprom_write_block(p, (uint8_t*)address, size);     // write EEPROM      
					
           		    msgLength = 2;
		            msgBuffer[1] = STATUS_CMD_OK;    		        
                }
                break;
                
            case CMD_READ_FLASH_ISP:
            case CMD_READ_EEPROM_ISP:                                                
                {
                    unsigned int  size = ((msgBuffer[1])<<8) | msgBuffer[2];                    
                    unsigned char *p = msgBuffer+1;
                    msgLength = size+3;
                    
                    *p++ = STATUS_CMD_OK;                    
                    if (msgBuffer[0] == CMD_READ_FLASH_ISP )
                    {
                        unsigned int data;
						/* Read FLASH */
                        do 
						{                            
#if defined(RAMPZ)
                			data = pgm_read_word_far(address);
#else
        		        	data = pgm_read_word_near(address);
#endif
        			        *p++ = (unsigned char)data;         //LSB
        			        *p++ = (unsigned char)(data >> 8);	//MSB  
        			        address    += 2;  	 // Select next word in memory
        			        size -= 2;
                        }while (size);
                    }
                    else
          			    eeprom_read_block(p, (uint8_t*) address, size); // Read EEPROM 
					
                    *p++ = STATUS_CMD_OK;
                }
                break;
				
	        default:
	            msgLength = 2;   
	            msgBuffer[1] = STATUS_CMD_FAILED;
	            break;
	    }
		
        /* Now send answer message back */
	    sendchar(MESSAGE_START);     
	    checksum = MESSAGE_START^0;
	       
	    sendchar(seqNum);
	    checksum ^= seqNum;
	        
	    c = ((msgLength>>8)&0xFF);
	    sendchar(c);
	    checksum ^= c;
	      
	    c = msgLength & 0x00FF;
	    sendchar(c);
	    checksum ^= c;
	        
	    sendchar(TOKEN);
	    checksum ^= TOKEN;
        p = msgBuffer;
		
        while (msgLength)
        {                
            c = *p++;
            sendchar(c);
            checksum ^=c;
            msgLength--;               
        }                   
	    sendchar(checksum);	        
	    seqNum++;
	
		uint16_t j=0;
		while(!newDataAvailable())
		{
			if(j++>20000)
			{
				// Flash not yet programmed -> wait for new Data
				if(pgm_read_word(0x0000) == 0xFFFF)
					j=0;

				else
					avrReset();
			}
			// wait a little bit before checking again for new Data
			for(uint8_t i=0;i<50;i++) asm volatile("nop");
		}
		

	}//for
	
	/* Never Reach this */
    for(;;);
}

void avrReset(void)
{
	/* We don't need to reenable rww section because we generate a Watchdog Reset */
	//boot_rww_enable();	// enable application section
	
	// Statusbyte ist already set to 0, so we don't need to do this anymore
	// Now Reset AVR with Watchdog
	wdt_enable(WDTO_15MS);	// 15 ms Timeout
	while(1);	// Wait until Reset is generated
}

