/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * dcf77.c
 *
 * Copyright (c) 2009 Torsten Giese
 *
 * ATMEGA88 @ 8 MHz
 *
 * $Id$
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#include <inttypes.h>
#include <avr/io.h>
#include <util/delay.h>
#include <avr/pgmspace.h>

#include "main.h"
#include "dcf77.h"

#define DCF_PORT        PORTB
#define DCF_DDR         DDRB
#define DCF_PIN         PINB
#define DCF_BIT         7

#define GPO4_PORT       PORTD
#define GPO4_DDR        DDRD
#define GPO4_BIT        4

#define TRUE            1
#define FALSE           0

volatile typedef struct{
  uint8_t   PauseCounter;                     // Counter for pause length (+1 each 10 ms)
  uint8_t   BitCounter;                       // which Bit is currently transfered
  uint8_t   Parity;                           // must be even when checked
  uint8_t   BCDShifter;                       // Data will be received in BCD Code
  uint8_t   NewTime[6];                       // store Date & Time while receiving DCF signal
  uint8_t   NewTimeShifter;                   // used to identify which data is currently transfered
  uint8_t   Check;                            // TRUE if received Data are correct
} DCF_Struct;

static volatile DCF_Struct    DCF;
uint8_t enable_dcf77_ISR;                     // En- / Disable DCF77 examination
/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * dcf77: clear all dcf variables
 * Reset all used variables if an error occurs while receiving the DCF77 signal
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void
dcf77_reset(void)
{
  uint8_t i;
  DCF.Parity          = 0;                    // Count of high pulse must be even
  DCF.PauseCounter    = 0;                    // Count length of pause to identify High/Low Signal
  DCF.BCDShifter      = 0;                    // transform transfered data from BCD to decimal
  DCF.BitCounter      = 0;                    // Count all received pulses per stream
  DCF.NewTimeShifter  = 0;                    // used to identify which data is currently transfered
  DCF.Check           = FALSE;                // TRUE if received DCF77 data stream was valid
  for (i=0; i <= 7; i++)
  {
    DCF.NewTime[i]    = 0;                    // Store temporary new received data
  }
}
/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * dcf77: analyze received signal
 * Analyze the received Signal and calculate the reflect data
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static uint8_t
dcf77_check(void)
{
  uint8_t  BCD_Kodierung[8] =
  {
      (  1 ),  (  2 ),  (  4 ),  (  8 ),
      ( 10 ),  ( 20 ),  ( 40 ),  ( 80 )
  };
  if ((DCF.PauseCounter <= 6)                   // spike or
      || (DCF.PauseCounter >= 196)              // pause longer then sync or
      || ((DCF.PauseCounter <= 77)              // pause too short and
          && (DCF.PauseCounter >= 96)))         // pause too long for data
  {
    DCF.PauseCounter = 0;                       // clear pause length counter
    return(FALSE);
  }
  if (((DCF.PauseCounter >= 170)                // received sync pulse
      && (DCF.BitCounter != 58))                // but not 58 bits transfered
      || (DCF.BitCounter >= 59))                // or more then 58 Bits
  {
    dcf77_reset();                              //   then reset
    return(FALSE);
  }
  if (DCF.BitCounter <= 20)                     // Bit 0 - 20 = Weather data
  {
    DCF.BitCounter++;                           // increase bit counter
    DCF.PauseCounter = 0;                       // clear pause length counter
    return(FALSE);
  }
  if ((DCF.PauseCounter >= 78)
      && (DCF.PauseCounter <= 95))              // 0 or 1 detect?
  {
    if (DCF.PauseCounter <= 86)                 // 1 detect?
    {
      DCF.Parity++;                             // increase parity counter
      if (!((DCF.BitCounter == 28)              // not Minutes Parity or
          || (DCF.BitCounter == 35)))           // not Hours Parity?
      {
        DCF.NewTime[DCF.NewTimeShifter] +=
            BCD_Kodierung[DCF.BCDShifter];      // NewTime Value + BCD-Value
      }
    }
    DCF.BCDShifter++;                           // increase BCD Shifter for next data value
    if (((DCF.BitCounter == 28)                 // Minutes Parity or
      || (DCF.BitCounter == 35))                // Hours Parity and
          && (DCF.Parity%2 != 0))               // ParityCount not even?
    {
      dcf77_reset();                            //   then reset
      return(FALSE);
    }
    if ((DCF.BitCounter == 28)                  // next will be Hours
      || (DCF.BitCounter == 35)                 // next will be Day of Month
      || (DCF.BitCounter == 41)                 // next will be Day of Week
      || (DCF.BitCounter == 44)                 // next will be Month
      || (DCF.BitCounter == 49))                // next will be Year
    {
      DCF.NewTimeShifter++;                     // increase new time shifter
      DCF.BCDShifter = 0;                       // reset BCD Shifter for next data stream
    }
    DCF.BitCounter++;                           // increase Bit Counter
    DCF.PauseCounter = 0;                       // clear pause length counter
    return(FALSE);
  }
  if (DCF.PauseCounter >= 170)                  // sync pause (longer then 1700 ms)?
  {
    if (!(DCF.PauseCounter >= 187))             // last Bit = 1 ?
    {
      DCF.Parity++;                             //   then Parity + 1
    }
    if ((DCF.BitCounter != 58)                  // do not receive 58 bits or
      || (DCF.Parity%2 != 0))                   // ParityCount not even?
    {
      dcf77_reset();                            //   then reset
      return(FALSE);
    }
    return(TRUE);                               // Everything fine, received Data could be take over
  }
  return(FALSE);
}
/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * dcf77: Initialize Port
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
dcf77_init(void)
{
  DCF_DDR             &= ~ (1 << DCF_BIT);      // set DCF Pin as input
  DCF_PORT            &= ~ (1 << DCF_BIT);      // deactivate Pull-Up Resistor

  dcf77_reset();

  GPO4_DDR |= (1<<4);                           // set GPO4 to output
  GPO4_PORT &= ~(1<<4);                         // set GPO4 to low
}
/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * dcf77: interrupt routine to count the pause length
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
dcf77_ISR(void)
{
  if (enable_dcf77_ISR)                         // Must be TRUE to enable DCF77 analysis
  {
    if (!(DCF_PIN & (1 << DCF_BIT)))            // DCF77 receiver port HIGH? (low active)
    {
      DCF.PauseCounter++;                       // YES -> increment PauseCounter
      GPO4_PORT &= ~(1<<GPO4_BIT);              // deactivate GPO4
    }
    else
    {
      DCF.Check = TRUE;                         // NO -> Set DCF.Check (Pulse has ended)
      GPO4_PORT |= (1<<GPO4_BIT);               // activate GPO4
    }
  }
}
/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * dcf77: set datetime with dcf77 time
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
uint8_t
dcf77_getDateTime(DATETIME * DateTime_p)
{
  if (DCF.Check)                                // full pulse received?
  {                                             // YES ->
    if (dcf77_check())                          // received data are correct?
    {                                           // YES ->
      DateTime_p->mm  = DCF.NewTime[0];         // then take over the DCF-Time
      DateTime_p->hh  = DCF.NewTime[1];
      DateTime_p->DD  = DCF.NewTime[2];
      DateTime_p->wd  = DCF.NewTime[3];
      DateTime_p->MM  = DCF.NewTime[4];
      DateTime_p->YY  = DCF.NewTime[5];
      dcf77_reset();                            // Reset Variables
      enable_dcf77_ISR = FALSE;                 // Clear enable_dcf77_ISR
      GPO4_PORT &= ~(1<<GPO4_BIT);              // deactivate GPO4

      return (TRUE);
    }
    DCF.Check = FALSE;
  }
  return (FALSE);
}
/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * dcf77: END
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
