/*------------------------------------------------------------------------------------------------------------------------------------------------*//**
 * @file display.h
 * 
 *  The display-module is main interface to the display hardware of the clock.
 *  \details
 *    The display-module its to convert the given time to the binary data 
 *    to controll the status of the leds.\n
 *    This file should be left unchanged if making adaptions to other languages. \n
 *    The language specific things resides in display_[language].h/c
 *
 * $Id$
 * 
 * \author Vlad Tepesch    Copyright (c) 2009 
 * 
 * \remarks
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 */
 /*-----------------------------------------------------------------------------------------------------------------------------------------------*/

#ifndef _WC_DISPLAY_H_
#define _WC_DISPLAY_H_


/*
 * For language/front related constants look at the approriate display_[language].h
 */

#if defined   (WC_LANG_ENG)
#  include "display_eng.h"
#elif defined (WC_LANG_GER)
#  include "display_ger.h" 
#else   /* default to german */
#  define WC_LANG_GER
#  include "display_ger.h" 
#endif


#ifdef __cplusplus
extern "C"
{
#endif 


/**
 * This Enum defines how the led words are connected to the Board and the position 
 * in the state data (	that's why the minutes (gpio, not shift register) are also in this enum )
 * @details In different languages at least the constants for the 
 *          hours (DWP_one to DWP_twelve), the it-is (DWP_itis), the four minutes (DWP_min[1234]) and 
 *          clock (DWP_clock) have to exist, so they can be used for interface
 *          the minute words are too diffrent and should not be used outside of display_[language].h/c
 *          Following two preconditions were made:
 *          - the eight Minute words are placed consecutively
 *            (but not neccessaryly ordered)
 *            and DWP_MIN_FIRST defines the first of them
 *          - the twelve hours are orderd consecutively
 *            and DWP_HOUR_BEGIN defines the first of them
 */
typedef enum e_displayWordPos e_displayWordPos;

/** interval blinking words (* 100ms) */
#define DISPLAY_BLINK_INT_100ms 7

/** how fast the fading between two times will be */
#define DISPLAY_FADE_TIME_ms 1000


/* ports, pins and ddrs for minute LEDs*/
#define DISPLAY_MIN1_PORT PORTB
#define DISPLAY_MIN1_DDR  DDRB
#define DISPLAY_MIN1_PIN  PIN0

#define DISPLAY_MIN2_PORT PORTD
#define DISPLAY_MIN2_DDR  DDRD
#define DISPLAY_MIN2_PIN  PIN7

#define DISPLAY_MIN3_PORT PORTC
#define DISPLAY_MIN3_DDR  DDRC
#define DISPLAY_MIN3_PIN  PIN2

#define DISPLAY_MIN4_PORT PORTC
#define DISPLAY_MIN4_DDR  DDRC
#define DISPLAY_MIN4_PIN  PIN3


#define DISPLAY_TIMER_OCR      OCR2A
#define DISPLAY_TIMER_OVF_vect TIMER2_OVF_vect
#define DISPLAY_TIMER_OCR_vect TIMER2_COMPA_vect
#define DISPLAY_TIMER_ENABLE_INTS() do{ TIMSK2 |= (1<<TOIE2) | (1<<OCIE2A);}while(0)
#define F_TIMER2 488  /* 1/s */




/**
 * contains inforation that words should be displayed
 * @details  DisplayState is a bitset.
 *           The bitset is defined by the enum e_displayWordPos.
 */
typedef uint32_t DisplayState;


/**
 * contains display paraeters
 * Definition of struct has to be in language specific header.
 */
typedef struct DisplayEepromParams DisplayEepromParams;


/**
 *  global instance of display Params.
 *  @TODO: consider creating access functions
 */
extern DisplayEepromParams* g_displayParams;



/**
 *  initializes the display
 *  @details Configures display IO hardware and clears display
 */
extern void    display_init (void);




/**
 * calculates which display elements should be active to display the giben time
 * @details  f_call: extern irregularly  (internal: when called setNewTime)
 * @param    i_newDateTime  the new time that should be displayed
 * @return   DisplayState as needed by setDisplayState
 */
extern DisplayState display_getTimeState (const DATETIME* i_newDateTime);


/**
 *  sets the display to the given state
 *  @details This function can be used to set a special image on the display independent of current time
 *           This is especially usefull for provide some feesback for user interactions
 *           eg. set time, or set Color
 *           f_call: extern irregularly  (internal: when called setNewTime)
 *  @param    i_showStates   defined which words should be shown at all
 *  @param    i_blinkstates  defines which words should blink with DISPLAY_BLINK_INT_ms
 *                           only words that should be displayed will blink 
 *                           words_that_will_blink = blinkstates & showStates
 */
extern void    display_setDisplayState( DisplayState i_showStates, DisplayState i_blinkstates);



/**
 * sets a new time to the Display
 * @details  f_call = on time change ~1-2/min
 * @param i_newDateTime  the new time that should be displayed
 */
static inline void display_setNewTime (const DATETIME* i_newDateTime)
{
  display_setDisplayState( display_getTimeState(i_newDateTime), 0 );
};



/**
 * fades out the old and fades in the new time to the Display
 * @details  f_call = on time change ~1-2/min
 * @param i_newDateTime  the new time that should be displayed
 */
void display_fadeNewTime (const DATETIME* i_newDateTime);




/**
 * Updates the display if currently in a blinking-state state, else does nothing
 * @details  have to be called from 10Hz ISR
 *           f_call: 10/s
 */
extern void    display_blinkStep (void);


#ifdef __cplusplus
}
#endif 


#endif /* _WC_DISPLAY_H_ */