/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * ldr.c
 *
 * Copyright (c) 2009 Ren Harsch ( rene@harsch.net )
 *
 * ATMEGA88 @ 8 MHz
 *
 * $Id$
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

#include <inttypes.h>
#include <avr/io.h>
#include <util/delay.h>
#include <avr/pgmspace.h>
#include <avr/interrupt.h>

#include "main.h"
#include "ldr.h"

static volatile uint8_t             array[LDR_ARRAY_SIZE];
static volatile uint16_t            curr_sum;


/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * adc isr
 * recalculate our new summary
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
/* later
ISR(ADC_vect)
{


}
*/

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * ldr_init: initialize adc for ldr
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
ldr_init (void)
{
  volatile uint8_t result;
 
  ADMUX = 0;                        // ADC0
  ADMUX |= (1<<REFS0);              // use Avcc
  ADMUX |= ADLAR;
 
  ADCSRA = (1<<ADEN) | (1<<ADPS2) | (1<<ADPS0);    	// prescaler to 32 / in our case sampling frequence of 250kHz
                                                   	// and activate ADC

  ADCSRA |= (1<<ADSC);              				// ADC start 
  while ( ADCSRA & (1<<ADSC) ) {
    ;     
  }

  result = ADCH;  // read out the value and init the summary array
  for (int i = 0; i < LDR_ARRAY_SIZE; array[i++] = result);

  curr_sum = LDR_ARRAY_SIZE * result;


  // later enable ADC interrupt
  // ADCSRA |= (1 << ADIE); 

  return;
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * ldr_get_brightness: returns the average of brightness 
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
uint8_t
ldr_get_brightness (void)
{
	return ( 255-(curr_sum / LDR_ARRAY_SIZE)); 
}


/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * ldr isr, called every 1 sec
 * recalculate our new summary and start a new conversion
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
ldr_ISR (void)
{
    static uint8_t   curr_index = 0;

	if (ADCSRA & (1<<ADSC)) {
		// read out last conversion and recalculating summary
		curr_sum -= array[curr_index];
		array[curr_index] = ADCH;
		curr_sum += array[curr_index++];

    if (curr_index >= LDR_ARRAY_SIZE) curr_index = 0;

		// start next ADC converting
		ADCSRA |= (1<<ADSC); 
	} 

	return;
}
