/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * pwm.c
 *
 * Copyright (c) 2009 Frank Meyer - frank(at)fli4l.de
 *
 * ATMEGA88 @ 8 MHz
 *
 * $Id$
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

#include <inttypes.h>
#include <avr/io.h>
#include <util/delay.h>
#include <avr/pgmspace.h>

#include "main.h"
#include "pwm.h"

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * PWM: constants/variables
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#define PWM_RED_PORT          PORTD                                             // port D
#define PWM_RED_DDR           DDRD                                              // ddr D
#define PWM_RED_BIT           6                                                 // OC0A

#define PWM_GREEN_PORT        PORTD                                             // port D
#define PWM_GREEN_DDR         DDRD                                              // ddr D
#define PWM_GREEN_BIT         5                                                 // OC0B

#define PWM_BLUE_PORT         PORTD                                             // port D
#define PWM_BLUE_DDR          DDRD                                              // ddr D
#define PWM_BLUE_BIT          3                                                 // OC2B

uint8_t pwm_table[MAX_PWM_STEPS]  PROGMEM =
{
  0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 16, 19, 23, 27, 32, 38, 45, 54, 64, 76, 91, 108, 128, 152, 181, 215, 255
};

static uint8_t                offset_pwm_idx;                                   // current offset pwm index
static uint8_t                base_pwm_idx;                                     // current base pwm index
static uint8_t                red_pwm_idx;                                      // current red pwm index
static uint8_t                green_pwm_idx;                                    // current green pwm index
static uint8_t                blue_pwm_idx;                                     // current blue pwm index
static uint8_t                pwm_is_on;                                        // flag: pwm is on


/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Set brightness by step
 *  @details  sets brightness by global step values base_pwm_idx + offset_pwm_idx
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void
pwm_set_brightness_step (void)
{
  uint8_t  pwm_idx = base_pwm_idx + offset_pwm_idx;
  uint8_t  pwm_value;

  if (pwm_idx >= MAX_PWM_STEPS)
  {
    pwm_idx = MAX_PWM_STEPS - 1;
  }

  pwm_value = pgm_read_byte (pwm_table + pwm_idx);

  OCR0A  = pwm_value;
  OCR0B  = pwm_value;
  OCR2B  = pwm_value;
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Initialize the PWM
 *  @details  Configures 0CR0A, 0CR0B and 0CR2B as PWM channels
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
pwm_init (void)
{
  PWM_RED_PORT &= ~(1<<PWM_RED_BIT);                                            // set PWM_RED_BIT to low
  PWM_RED_DDR |= (1<<PWM_RED_BIT);                                              // set PWM_RED_BIT to output

  PWM_GREEN_PORT &= ~(1<<PWM_GREEN_BIT);                                        // set PWM_RED_BIT to low
  PWM_GREEN_DDR |= (1<<PWM_GREEN_BIT);                                          // set PWM_RED_BIT to output

  PWM_BLUE_PORT &= ~(1<<PWM_BLUE_BIT);                                          // set PWM_BLUE_BIT to low
  PWM_BLUE_DDR |= (1<<PWM_BLUE_BIT);                                            // set PWM_BLUE_BIT to output

  TCCR0A = (1<<WGM01) | (1<<WGM00);                                             // non-inverted PWM, 8 Bit Fast PWM

  //                                                                            // values for Fast PWM:
  // TCCR0B = (1<<CS00);                                                        // 001: prescaler    1 -> 8 MHz / 256 /    1 = 31250 Hz PWM frequency
  // TCCR0B = (1<<CS01);                                                        // 010: prescaler    8 -> 8 MHz / 256 /    8 =  3906 Hz PWM frequency
  // TCCR0B = (1<<CS01) | (1<<CS00);                                            // 011: prescaler   64 -> 8 MHz / 256 /   64 =   488 Hz PWM frequency
  // TCCR0B = (1<<CS02);                                                        // 100: prescaler  256 -> 8 MHz / 256 /  256 =   122 Hz PWM frequency
  // TCCR0B = (1<<CS02) | (1<<CS00);                                            // 101: prescaler 1024 -> 8 MHz / 256 / 1024 =    31 Hz PWM frequency

  TCCR0B = (1<<CS01) | (1<<CS00);                                               // 011: prescaler   64 -> 8 MHz / 256 /   64 =   488 Hz PWM frequency

  TCCR2A = (1<<WGM21) | (1<<WGM20);                                             // non-inverted PWM, 8 Bit Fast PWM

  //                                                                            // values for Fast PWM:
  // TCCR2B = (1<<CS20);                                                        // 001: prescaler    1 -> 8 MHz / 256 /    1 = 31250 Hz PWM frequency
  // TCCR2B = (1<<CS21);                                                        // 010: prescaler    8 -> 8 MHz / 256 /    8 =  3906 Hz PWM frequency
  // TCCR2B = (1<<CS21) | (1<<CS20);                                            // 011: prescaler   32 -> 8 MHz / 256 /   32 =   977 Hz PWM frequency
  // TCCR2B = (1<<CS22);                                                        // 100: prescaler   64 -> 8 MHz / 256 /   64 =   488 Hz PWM frequency
  // TCCR2B = (1<<CS22) | (1<<CS20);                                            // 101: prescaler  128 -> 8 MHz / 256 /  128 =   244 Hz PWM frequency
  // TCCR2B = (1<<CS22) | (1<<CS21);                                            // 110: prescaler  256 -> 8 MHz / 256 /  256 =   122 Hz PWM frequency
  // TCCR2B = (1<<CS22) | (1<<CS21) | | (1<<CS20);                              // 111: prescaler 1024 -> 8 MHz / 256 / 1024 =    31 Hz PWM frequency

  TCCR2B = (1<<CS22);                                                           // 100: prescaler   64 -> 8 MHz / 256 /   64 =   488 Hz PWM frequency
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Switch PWM on
 *  @details  Switches PWM on with a narrow spike on all 3 channels -> leds glowing
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
pwm_on (void)
{
#if 0
  OCR0A = 0;                                                                    // pwm value: 0 -> narrow spike
  OCR0B = 0;                                                                    // pwm value: 0 -> narrow spike
  OCR2B = 0;                                                                    // pwm value: 0 -> narrow spike
#endif

  TCCR0A |= (1<<COM0A1);                                                        // non-inverted PWM on OC0A, 8 Bit Fast PWM
  TCCR0A |= (1<<COM0B1);                                                        // non-inverted PWM on OC0B, 8 Bit Fast PWM
  TCCR2A |= (1<<COM2B1);                                                        // non-inverted PWM on OC2B, 8 Bit Fast PWM

  pwm_is_on = TRUE;                                                             // set flag
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Switch PWM off
 *  @details  Switches PWM off
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
pwm_off (void)
{
#if 0
  OCR0A  = 0;                                                                   // reset OCR0A
  OCR0B  = 0;                                                                   // reset OCR0B
  OCR2B  = 0;                                                                   // reset OCR2B
#endif

  TCCR0A &= ~(1<<COM0A1);                                                       // reset TCCR0A
  TCCR0A &= ~(1<<COM0B1);                                                       // reset TCCR0A
  TCCR2A &= ~(1<<COM2B1);                                                       // reset TCCR2A

  PWM_RED_PORT  &= ~(1<<PWM_RED_BIT);                                           // set PWM_RED_BIT to low
  PWM_GREEN_PORT  &= ~(1<<PWM_GREEN_BIT);                                       // set PWM_GREEN_BIT to low
  PWM_BLUE_PORT  &= ~(1<<PWM_BLUE_BIT);                                         // set PWM_BLUE_BIT to low

  pwm_is_on = FALSE;                                                            // reset flag
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Toggle PWM off/on
 *  @details  Toggles PWM off/on
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
pwm_on_off (void)
{
  if (pwm_is_on)
  {
    pwm_off ();
  }
  else
  {
    pwm_on ();
    pwm_set_brightness_step ();
  }
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Set RGB colors
 *  @details  sets RGB color
 *  @param    red: range 0-255
 *  @param    green: range 0-255
 *  @param    blue: range 0-255
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
pwm_set_colors (uint8_t red, uint8_t green, uint8_t blue)
{
  OCR0A  = red;
  OCR0B  = green;
  OCR2B  = blue;
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Get RGB colors
 *  @details  gets RGB color
 *  @param    pointer to value of red pwm: range 0-255
 *  @param    pointer to value of green pwm: range 0-255
 *  @param    pointer to value of blue pwm: range 0-255
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
pwm_get_colors (uint8_t * redp, uint8_t * greenp, uint8_t * bluep)
{
  *redp = OCR0A;
  *greenp = OCR0B;
  *bluep = OCR2B;
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Set RGB colors by step values
 *  @details  sets RGB color by step values
 *  @param    red_step: range 0 to MAX_PWM_STEPS
 *  @param    green_step: range 0 to MAX_PWM_STEPS
 *  @param    blue_step: range 0 to MAX_PWM_STEPS
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
pwm_set_color_step (uint8_t red_step, uint8_t green_step, uint8_t blue_step)
{
  if (red_step >= MAX_PWM_STEPS)
  {
    red_step = MAX_PWM_STEPS - 1;
  }

  if (green_step >= MAX_PWM_STEPS)
  {
    green_step = MAX_PWM_STEPS - 1;
  }

  if (blue_step >= MAX_PWM_STEPS)
  {
    blue_step = MAX_PWM_STEPS - 1;
  }

  red_pwm_idx   = red_step;
  green_pwm_idx = green_step;
  blue_pwm_idx  = blue_step;

  pwm_set_colors (pgm_read_byte (pwm_table + red_step), pgm_read_byte (pwm_table + green_step), pgm_read_byte (pwm_table + blue_step));
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Get RGB color step values
 *  @details  grts RGB color step values
 *  @param    pointer zo red_step: range 0 to MAX_PWM_STEPS
 *  @param    pointer zo green_step: range 0 to MAX_PWM_STEPS
 *  @param    pointer zo blue_step: range 0 to MAX_PWM_STEPS
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
pwm_get_color_step (uint8_t * red_stepp, uint8_t * green_stepp, uint8_t * blue_stepp)
{
  *red_stepp    = red_pwm_idx;
  *green_stepp  = green_pwm_idx;
  *blue_stepp   = blue_pwm_idx;
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Set base brightness by step 0-31
 *  @details  sets base brightness by step 0-31
 *  @param    pwm step 0-31
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
pwm_set_base_brightness_step (uint8_t pwm_idx)
{
  if (pwm_idx >= MAX_PWM_STEPS)
  {
    base_pwm_idx = MAX_PWM_STEPS - 1;
  }
  else
  {
    base_pwm_idx = pwm_idx;
  }

  if (pwm_is_on && base_pwm_idx + offset_pwm_idx < MAX_PWM_STEPS)
  {
    pwm_set_brightness_step ();
  }
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Step up brightness
 *  @details  Steps up brightness
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
pwm_step_up_brightness (void)
{
  if (pwm_is_on && base_pwm_idx + offset_pwm_idx + 1 < MAX_PWM_STEPS)
  {
    offset_pwm_idx++;
    pwm_set_brightness_step ();
  }
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Step down brightness
 *  @details  Steps down brightness
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
pwm_step_down_brightness (void)
{
  if (pwm_is_on && base_pwm_idx + offset_pwm_idx > 0)
  {
    offset_pwm_idx--;
    pwm_set_brightness_step ();
  }
}
