/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * timer.c
 *
 * Copyright (c) 2009 Frank Meyer - frank(at)fli4l.de
 *
 * ATMEGA88 @ 8 MHz
 *
 * $Id$
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

#include <inttypes.h>
#include <avr/io.h>
#include <util/delay.h>
#include <avr/pgmspace.h>
#include <avr/interrupt.h>

#include "main.h"
#include "timer.h"
#include "dcf77.h"
#include "irmp.h"
#include "ldr.h"
#include "user.h"
#include "display.h"

#include "uart.h"
#include "wceeprom.h"

#if (IR_LOGGING==1)

  static void logIr(uint8_t val)
  {
    static const uint8_t  c_startcycles =   5;  /**< min count of zeros before start of logging */
#   define                c_datalen       500   /**< loggin buffer size */
    static const uint8_t  c_endBits     = 200;  /**< loggin buffer size */
    static uint8_t s_data[c_datalen]; /**< logging buffer */ 
    static uint16_t s_dataIdx;     /**< number of written bits */
    static uint8_t  s_startcycles  /**< current number of start-zeros */;
    static char s_ctr;             /**< counts sequenced highbits - to detect end*/

    if( (val==0) && (s_startcycles < c_startcycles) && !s_dataIdx ) /* prevent that single random zeros init logging*/
    {
      ++s_startcycles;
    }else
    {
      s_startcycles = 0;

      if(    (val==0)                          /* start or continue logging on zero */
          || ((val == 1)&& (s_dataIdx != 0) )  /* one cannot init logging */
         )
      {
        if(val){                                          /* set or clear bit in bitarray */
          s_data[(s_dataIdx/8)] |= (1<<(s_dataIdx%8));
        }else{
          s_data[(s_dataIdx>>3)] &= ~(1<<(s_dataIdx%8));
        }
        ++s_dataIdx;
        if(val){                                          /* if high receifed then look at log-stoo condition */
          ++s_ctr;
          if(s_ctr > c_endBits){                          /* if stop condition (200 sequenced ones) meet then output on uart*/
            uint16_t i=0;
            for(;i< s_startcycles; ++i){
                uart_putc( '0' );                         /* the four ignored zeros */
            }
            for(i=0;i < (s_dataIdx-c_endBits+20)/8; ++i)  /* transform bitset into uart chars */
            {
              uint8_t d = s_data[i];
              uint8_t j=0;
              for(;j<8;++j){
                uart_putc( (d & 1)+'0' );
                d>>=1;
              }
            }
            uart_putc('\n');
            s_dataIdx = 0;
          }
        }else{
          s_ctr = 0;
        }
      }
    }

  }
#  define logIsr() do{logIr( (PINB & (1<<6))?1:0);}while(0)
#else
#  define logIsr()
#endif

#define F_INTERRUPT           10000                                             // frequency of interrupts

#define  INTERRUPT_10000HZ    {  logIsr(); ir_ISR(); }
#define  INTERRUPT_1000HZ     { }
#define  INTERRUPT_100HZ      { dcf77_ISR(); }
#define  INTERRUPT_10HZ       { user_isr10Hz(); display_blinkStep(); }
#define  INTERRUPT_1HZ        { main_ISR(); ldr_ISR(); user_isr1Hz(); }
#define  INTERRUPT_1M         { }

void
timer_init (void)
{
  OCR1A   = (F_CPU / F_INTERRUPT) - 1;                                          // compare value: 1/10000 of CPU frequency
  TCCR1B  = (1 << WGM12) | (1 << CS10);                                         // switch CTC Mode on, set prescaler to 1
  TIMSK1  = 1 << OCIE1A;                                                        // OCIE1A: Interrupt by timer compare (use TIMSK for ATMEGA162)
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * timer 1 compare handler, called every 1/10000 sec
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
ISR(TIMER1_COMPA_vect)
{
  static uint8_t  thousands_counter;
  static uint8_t  hundreds_counter;
  static uint8_t  tenths_counter;
  static uint8_t  seconds_counter;
  static uint8_t  minutes_counter;

  INTERRUPT_10000HZ;

  thousands_counter++;

  if (thousands_counter != 10)
  {
    return;
  }

  thousands_counter = 0;

  INTERRUPT_1000HZ;

  hundreds_counter++;

  if (hundreds_counter != 10)
  {
    return;
  }

  hundreds_counter = 0;

  INTERRUPT_100HZ;

  tenths_counter++;

  if (tenths_counter != 10)                                                     // generate 10Hz ....
  {
    return;
  }

  tenths_counter = 0;

  INTERRUPT_10HZ;

  seconds_counter++;

  if (seconds_counter != 10)                                                    // generate 1Hz ....
  {
    return;
  }

  seconds_counter = 0;

  INTERRUPT_1HZ;

  minutes_counter++;

  if (minutes_counter != 60)                                                    // generate 1/60 Hz ....
  {
    return;
  }

  minutes_counter = 0;

  INTERRUPT_1M;
}
