/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * uart.c
 *
 * Copyright (c) 2009 Frank Meyer - frank(at)fli4l.de
 *
 * ATMEGA88 @ 8 MHz
 *
 * $Id$
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

#include <inttypes.h>
#include <avr/io.h>
#include <avr/pgmspace.h>

#include "uart.h"

#define UART_BAUD             9600L

// calculate real baud rate:
#define UBRR_VAL              ((F_CPU+UART_BAUD*8)/(UART_BAUD*16)-1)            // round
#define BAUD_REAL             (F_CPU/(16*(UBRR_VAL+1)))                         // real baudrate
#define BAUD_ERROR            ((BAUD_REAL*1000)/UART_BAUD-1000)                 // error in promille

#if ((BAUD_ERROR > 10) || (-BAUD_ERROR < 10))
#  error Error of baud rate of RS232 UARTx is more than 1%. That is too high!
#endif

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Initialize  UART
 *  @details  Initializes UART
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
uart_init (void)
{
  UCSR0B |= (1<<TXEN0);                                                         // activate UART0 TX
  UBRR0H = UBRR_VAL >> 8;                                                       // store baudrate (upper byte)
  UBRR0L = UBRR_VAL & 0xFF;                                                     // store baudrate (lower byte)
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Send character
 *  @details  Sends character
 *  @param    ch character to be transmitted
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
uart_putc (unsigned char ch)
{
  while (!(UCSR0A & (1<<UDRE0)))
  {
    ;
  }

  UDR0 = ch;
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Send string
 *  @details  Sends '\0'-terminated string
 *  @param    s string to be transmitted
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
uart_puts (const char * s)
{
  while (*s)
  {
    uart_putc (*s++);
  }
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Send string (stored in program memory)
 *  @details  Sends String stored in program memory (e.g. flash memory)
 *  @param    s program memory string to be transmitted
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
uart_puts_p (const char * progmem_s)
{
  char ch;

  while ((ch = pgm_read_byte (progmem_s++)) != '\0')
  {
    uart_putc (ch);
  }
}
