/*------------------------------------------------------------------------------------------------------------------------------------------------*//**
 * @file user.c
 * 
 *  This file contains implementation of the user interface.
 *
 * $Id$
 * 
 * \author Vlad Tepesch    Copyright (c) 2009 
 * 
 * \remarks
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 */
 /*-----------------------------------------------------------------------------------------------------------------------------------------------*/


#include <inttypes.h>

#include "main.h"
#include "user.h"
#include "irmp.h"
#include "pwm.h"
#include "i2c-rtc.h"
#include "display.h"
#include "dcf77.h"

#include "base.h"
#include "wceeprom.h"
#include "uart.h"

typedef enum MenuStates{

  MS_waitForIrTrain  = 0,
  MS_changeR,
  MS_changeG,
  MS_changeB,
  MS_changeHue,

  MS_setTimeHour,
  MS_setTimeMin,

  MS_irTrain,
  MS_hueMode,
  MS_demoMode,
  MS_normalMode,

}MenuStates;


//#define HUE_STEPS MAX_PWM_STEPS
#define HUE_STEPS 256

#define HUE_MAX (HUE_STEPS * 6)
#if (HUE_MAX > 255)
  typedef uint16_t Hue_t;
#  define HUE_MANUAL_STEPS 10
#  define SetPWMs pwm_set_colors
#else
  typedef uint8_t Hue_t;
#  define SetPWMs pwm_set_color_step
#  define HUE_MANUAL_STEPS 1
#endif

static volatile MenuStates g_curState = MS_waitForIrTrain;

static uint8_t  g_keyDelay;
static volatile Hue_t    g_curHue;
static DATETIME g_dateTime;
static UserEepromParams* g_params;
static uint8_t  g_curColorProfile;
static uint8_t  g_prescale10Hz;

static uint8_t  g_secs;



static inline void setTimeMode();

static void  handleUpDown(int8_t dir);

static inline void trainIR( const IRMP_DATA* i_irCode );
static  void dispInternalTime(uint32_t blinkmask);
static void switchNormalMode(void);

#define MIN_LEDS_MASK ((1L<<DWP_min1)|(1L<<DWP_min2)|(1L<<DWP_min3)|(1L<<DWP_min4))


#if (USER_LOG_STATE == 1)
#  define   log_state(x)  uart_puts_P(x)
#else
#  define   log_state(x)  
#endif

#if (USER_LOG_IR_TRAIN == 1)
#  define   log_irTrain(x)  uart_puts_P(x)
#else
#  define   log_irTrain(x)  
#endif

#if (USER_LOG_TIME== 1)
#  define   log_time(x)  uart_puts(x)
void putTime()
{
  char txt[8];
  byteToStrLessHundred(g_dateTime.hh, txt);
  uart_puts(txt);
  uart_putc(':');
  byteToStrLessHundred(g_dateTime.mm, txt);
  uart_puts(txt);
  uart_putc('\n');
}
#else
#  define   log_time(x)  
#  define   putTime()
#endif

/**
 *  generates the waveform for the green color
 */
static uint8_t hueWaveform(Hue_t x)
{
  if(x < HUE_MAX/6){
    return x*((HUE_STEPS*6)/HUE_MAX);
  }else if(x< (HUE_MAX*3)/6){
    return HUE_STEPS-1;
  }else if(x< (HUE_MAX*4)/6){
    return (((HUE_MAX*4)/6)-1-x)*((HUE_STEPS*6)/HUE_MAX);
  }else{
    return 0;
  }
}


/**
 * generates rgb from hue with saturation 1 and brightness 1
 */
static void hue2rgb(
              Hue_t h, /*uint8  s, uint8  v,*/
              uint8_t* r, uint8_t* g, uint8_t* b
              /* ,sint32 relsat= -1 */ )
{
  uint16_t barg = (((uint16_t)h)+2*HUE_MAX/3);
  uint16_t rarg = (((uint16_t)h)+HUE_MAX/3);

  if(barg>=HUE_MAX){
    barg-=HUE_MAX;
  }
  if(rarg>=HUE_MAX){
    rarg-=HUE_MAX;
  }

  *g = hueWaveform(h);
  *b = hueWaveform( (Hue_t)barg );
  *r = hueWaveform( (Hue_t)rarg );

  //{
  //  char buff[7];
  //  uart_puts_P("Hue: ");
  //  uint16ToHexStr(h, buff);
  //  uart_puts(buff);
  //  uart_putc(' ');
  //  byteToStr(*r, buff);
  //  uart_puts(buff);
  //  uart_putc(' ');
  //  byteToStr(*g, buff);
  //  uart_puts(buff);
  //  uart_putc(' ');
  //  byteToStr(*b, buff);
  //  uart_puts(buff);
  //  uart_putc('\n');
  //}

}





void user_init(void)
{
  //uint16_t col;
  g_params = &wcEeprom_getData()->userParams;

  //col = g_params->colorPresets[g_curColorProfile];
  //pwm_set_color_step( col & 0x1F, (col>>5) & 0x1F, (col>>10) & 0x1F);

  display_setDisplayState( MIN_LEDS_MASK, MIN_LEDS_MASK );
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * USER: handle RC5
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
handle_ir_code (void)
{
  IRMP_DATA ir_data;

  if (ir_get_data (&ir_data))
  {
    if(g_keyDelay){
      return;
    }
#   if (USER_LOG_IR_CMD == 1)
    {
      char text[20];
      uart_puts_P("IR-cmd: ");
      uint16ToHexStr(ir_data.protocol, text);
      uart_puts(text);
      uint16ToHexStr(ir_data.address, text);
      uart_puts(text);
      uint16ToHexStr(ir_data.command, text);
      uart_puts(text);
      uart_putc('\n');
    }
#   endif
    g_keyDelay = USER_KEY_PRESS_DELAY_100ms;
    if(    (g_curState == MS_waitForIrTrain)
        || (g_curState == MS_irTrain)
       ){
      g_curState = MS_irTrain;
      trainIR( &ir_data );
    }else{
        uint8_t ir_code;
        uint8_t idx = 0;
        if( g_params->irAddress != ir_data.address )
          return;
        while(    (idx<UI_COMMAND_COUNT)
               && (g_params->irCommandCodes[idx] != ir_data.command)){
                 ++idx;
        }
        ir_code = idx;

        if( UI_ONOFF == ir_code ){  /* has to be first because keys that the user does not need will have ONOFF code */
            log_state("OF\n");
            pwm_on_off ();
        }else if( UI_BRIGHTNESS_UP == ir_code ){
            log_state("B+\n");
            pwm_step_up_brightness ();
        }else if( UI_BRIGHTNESS_DOWN == ir_code ){
            log_state("B-\n");
            pwm_step_down_brightness ();
        }else if(    (UI_UP == ir_code )
                  || (UI_DOWN == ir_code )
                ){
            handleUpDown( (UI_UP == ir_code)?1:-1  );
        }else if(    (UI_CHANGE_R == ir_code)
                  || (UI_CHANGE_G == ir_code)
                  || (UI_CHANGE_B == ir_code)
                  || (UI_CHANGE_HUE == ir_code)
                )
        {
            if(UI_CHANGE_R == ir_code){    /* if logging is inactive the whole will be dropped by optimizer*/
              log_state("CR\n");
            }else if(UI_CHANGE_G == ir_code){
              log_state("CG\n");
            }else if(UI_CHANGE_B == ir_code){
              log_state("CB\n");
            }else if(UI_CHANGE_HUE == ir_code){
              log_state("CH\n");
            }
            g_curState = MS_changeR + idx-UI_CHANGE_R;
        }else if( UI_SET_TIME == ir_code ){
            setTimeMode();
        }else if( UI_NORMAL_MODE == ir_code ){
          switchNormalMode();
        }else if( UI_HUE_MODE == ir_code ){
            log_state("HM\n");
            g_prescale10Hz = USER_HUE_CHANGE_INT_100ms;
            g_curState = MS_hueMode;
        }else if( UI_DEMO_MODE == ir_code ){
            log_state("DM\n");
            g_prescale10Hz = USER_DEMO_CHANGE_INT_100ms;
            g_curState = MS_demoMode;
    #ifdef WC_LANG_GER
        }else if( UI_TOGGLE_OSSI == ir_code ){
            g_displayParams->ossiMode = !g_displayParams->ossiMode;
            log_state("WO\n");
            user_setNewTime( 0 );
    #endif
        }else if( UI_DCF_GET_TIME == ir_code ){
            enable_dcf77_ISR = TRUE;
    //    }else if( g_params->irCommandCodes[] == ir_code ){
        }else{
          return;
        }
    }
    g_secs = 0; /* on recognized keypress rest the counter */
  }
}


static void setTimeMode()
{
  if(g_curState==MS_setTimeHour){
    log_time("TM\n");
    g_curState = MS_setTimeMin;
    dispInternalTime( DISPLAY_MINUTE_MASK | (1L<<DWP_clock));
  }else if( g_curState==MS_setTimeMin ){
    log_time("TS\n");
    g_curState = MS_normalMode;
    g_dateTime.ss = 0;
    i2c_rtc_write( &g_dateTime );
  }else{
    log_time("TH\n");
    g_curState = MS_setTimeHour;
    dispInternalTime( DISPLAY_HOUR_MASK | (1L<<DWP_clock) );
  }

}

static void trainIR( const IRMP_DATA* i_irCode )
{
  static uint8_t s_curKey = 0;
  uint8_t  hourTodisp;

  if( s_curKey>0){  // save key
    if( g_params->irAddress != i_irCode->address ){
      log_irTrain("invalid ir-address\n");
      return;
    }
    g_params->irCommandCodes[s_curKey-1] = i_irCode->command;
    if( s_curKey == UI_COMMAND_COUNT ){ /* finished */
      g_curState = MS_normalMode;
      log_irTrain("Ir train finished\n");
      wcEeprom_writeback( &g_params, sizeof(*g_params) ); /* save all */
      return;
    }
  }else{
    g_params->irAddress = i_irCode->address;
  }

# if USER_LOG_IR_TRAIN
  {
    char buff[5];
    uart_puts_P("Ir train. Enter cmd #");
    byteToStrLessHundred(s_curKey, buff);
    uart_puts(buff);
    uart_putc('\n');
  }
# endif
  hourTodisp = s_curKey%12;

  display_setDisplayState(
      (1L << (hourTodisp + DWP_HOUR_BEGIN)) | MIN_LEDS_MASK ,
      (1L << (hourTodisp + DWP_HOUR_BEGIN)) | MIN_LEDS_MASK);

  ++s_curKey;
}

static uint8_t incDecRange(uint8_t val, int8_t dir,  uint8_t max)
{
  val += dir;
  if(val == 255){
    val = max;
  }else if(val>max){
    val = 0;
  }
  return val;
}

static void
handleUpDown(int8_t dir)
{
  switch(g_curState){
    case MS_changeR:
    case MS_changeG:
    case MS_changeB:
    {
      uint8_t rgb[3];
      pwm_get_color_step( &rgb[0], &rgb[1], &rgb[2]);
      if(    (dir>0) 
          || (rgb[g_curState-MS_changeR] > 0))
      {
        rgb[g_curState-MS_changeR] += dir;
      }
      pwm_set_color_step(rgb[0], rgb[1], rgb[2]);
      g_params->colorPresets[g_curColorProfile] = rgb[0] | (rgb[1]<<5) | (rgb[2]<<10);
      break;
    }
    case MS_changeHue:
    {
      uint8_t r;
      uint8_t g;
      uint8_t b;

      if(    (dir < 0) )
      { 
        if( (g_curHue>HUE_MANUAL_STEPS) ){
          g_curHue -= HUE_MANUAL_STEPS;
        }else{
          g_curHue = HUE_MAX;
        }
      }else{
        g_curHue += HUE_MANUAL_STEPS;
        g_curHue %= HUE_MAX;
      }

      hue2rgb(g_curHue, &r, &g, &b);
      SetPWMs(r,g,b);
      break;
    }
    case MS_setTimeMin:
      g_dateTime.mm = incDecRange(g_dateTime.mm, dir, 59);
      dispInternalTime( DISPLAY_MINUTE_MASK  | (1L<<DWP_clock) );
      break;
    case MS_setTimeHour:
      g_dateTime.hh = incDecRange(g_dateTime.hh, dir, 23);
      dispInternalTime(DISPLAY_HOUR_MASK  | (1L<<DWP_clock) );
      break;
    default: ;
  }
}


static void dispInternalTime(uint32_t blinkmask)
{
  putTime();
  display_setDisplayState( display_getTimeState( &g_dateTime ), blinkmask);
}



extern void  user_setNewTime( const DATETIME* i_time)
{
  if(    (g_curState != MS_setTimeHour)
      && (g_curState != MS_setTimeMin) )
  {
    if( i_time ){
      log_time("saved Time ");
      g_dateTime = *i_time;
    }

    if(    (g_curState != MS_demoMode )
        && (g_curState != MS_irTrain )
        && (g_curState != MS_waitForIrTrain ) 
        && ((g_curState != MS_normalMode) || (g_prescale10Hz==0)) )
    {
      log_time("disp Time");
      putTime();
      display_fadeNewTime(&g_dateTime);
      //dispInternalTime( 0 );
    }
    log_time("\n");
  }
}

void switchNormalMode()
{
  uint16_t col;
  uint32_t dispData;
  log_state("NM\n");
  if(g_curState != MS_normalMode){
    g_curState = MS_normalMode;
  }else{
    ++g_curColorProfile;
    g_curColorProfile %=  UI_COLOR_PRESET_COUNT;

  }
  col = g_params->colorPresets[g_curColorProfile];
  pwm_set_color_step( col & 0x1F, (col>>5) & 0x1F, (col>>10) & 0x1F);
  dispData =  (1L<<(DWP_HOUR_BEGIN + g_curColorProfile));
# ifdef WC_LANG_GER
  if( g_curColorProfile == 0){
    dispData |= (1L<<DWP_s);
  }
# endif
  display_setDisplayState(dispData, 0);

  g_prescale10Hz = USER_NORMAL_SHOW_NUMBER_DELAY_100ms;
}




void user_isr10Hz(void)
{
  static MenuStates s_lastState = MS_waitForIrTrain;

  if( s_lastState != g_curState){
    user_setNewTime( 0 );
    s_lastState = g_curState;
  }

  if( g_keyDelay ){
    --g_keyDelay;
  }
  if(    (g_curState == MS_normalMode)
      && (g_prescale10Hz > 0) )
  {
    --g_prescale10Hz;
    if( g_prescale10Hz == 0 ){
      dispInternalTime(0);
    }

  }else if( g_curState == MS_hueMode )
  {
    if( ! (--g_prescale10Hz) )
    {
      uint8_t r;
      uint8_t g;
      uint8_t b;
      ++g_curHue;
      g_curHue %= (HUE_MAX+1);
      hue2rgb(g_curHue, &r, &g, &b);
      SetPWMs(r,g,b);
      g_prescale10Hz = USER_HUE_CHANGE_INT_100ms;
    }
  }else if( g_curState == MS_demoMode ){
    if( ! (--g_prescale10Hz) )
    {
      static uint8_t s_demoStep = 0;
      display_setDisplayState((1L << s_demoStep), 0);
      ++s_demoStep;
      s_demoStep %= 32;
      g_prescale10Hz = USER_DEMO_CHANGE_INT_100ms;
    }
  }
}


void  user_isr1Hz(void)
{
  ++g_secs;
  if( g_curState == MS_waitForIrTrain ){
    if( g_secs == USER_STARTUP_WAIT_4_IR_TRAIN_s ){
      log_irTrain("leave IR-wait4train\n");
      g_secs=0;
      switchNormalMode();
    }
  }else{
    if(g_secs == USER_DELAY_BEFORE_SAVE_EEPROM_s){
      wcEeprom_writeback( wcEeprom_getData(), sizeof(WcEepromData)); /* save whole data */
      g_secs = 0;
    }
  }
}

