/*------------------------------------------------------------------------------------------------------------------------------------------------*//**
 * @file user.h
 * 
 *  The user-module handles the user interface of the wordclock
 *  \details
 *    This includes the reaction on the input via IR-RC as well as
 *    the output of the time.
 *
 * $Id$
 * 
 * \author Vlad Tepesch    Copyright (c) 2009 
 * 
 * \remarks
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 */
 /*-----------------------------------------------------------------------------------------------------------------------------------------------*/


#ifndef _WC_USER_H_
#define _WC_USER_H_

#include "main.h"

#ifdef __cplusplus
extern  "C"
{
#endif

typedef enum e_userCommands
{
  UI_ONOFF           = 0,
  UI_BRIGHTNESS_UP   ,
  UI_BRIGHTNESS_DOWN ,

  UI_UP              ,
  UI_DOWN            ,

  UI_SET_TIME        ,
  UI_DCF_GET_TIME    ,

  UI_CHANGE_R        ,
  UI_CHANGE_G        ,
  UI_CHANGE_B        ,
  UI_CHANGE_HUE      ,

  UI_NORMAL_MODE     ,
  UI_HUE_MODE        ,
  UI_DEMO_MODE       ,
  UI_TOGGLE_OSSI     ,

  UI_COMMAND_COUNT
}e_userCommands;


#define USER_COMMANDCODES_DEFAULTS { \
 /* [UI_ONOFF          ] =  */  0x00c8  /* power */,  \
 /* [UI_BRIGHTNESS_UP  ] =  */  0x0018  /* fwd   */,  \
 /* [UI_BRIGHTNESS_DOWN] =  */  0x0098  /* bwd   */,  \
                                                      \
 /* [UI_UP             ] =  */  0x0050  /* ch+   */,  \
 /* [UI_DOWN           ] =  */  0x00d0  /* ch-   */,  \
                                                      \
 /* [UI_SET_TIME       ] =  */  0x0090  /* 9     */,  \
 /* [UI_DCF_GET_TIME   ] =  */  0x0000  /* 0     */,  \
                                                      \
 /* [UI_CHANGE_R       ] =  */  0x0020  /* 4     */,  \
 /* [UI_CHANGE_G       ] =  */  0x00a0  /* 5     */,  \
 /* [UI_CHANGE_B       ] =  */  0x0060  /* 6     */,  \
 /* [UI_CHANGE_HUE     ] =  */  0x00e0  /* 7     */,  \
                                                      \
 /* [UI_NORMAL_MODE    ] =  */  0x0080  /* 1     */,  \
 /* [UI_HUE_MODE       ] =  */  0x0040  /* 2     */,  \
 /* [UI_DEMO_MODE      ] =  */  0x00c0  /* 3     */,  \
 /* [UI_TOGGLE_OSSI    ] =  */  0x0010  /* 8     */,  \
}


/** the maximum number of color-presets that can be stored */
#define UI_COLOR_PRESET_COUNT 4

#define USER_COLORPRESETS_DEFAULTS {\
    31 /* red   */, \
   992 /* green */, \
 31744 /* blue  */, \
 32767 /* white */, \
}

#define USER_ADDRESS_DEFAULT 0x01DE
#define USER_DUMMY_COMMAND   0xffFF

/** contains the persistent data of the user module that should be stored in eeprom */
typedef struct UserEepromParams{

  /** Address of the trained remote control */
  uint16_t irAddress;

  /**
   *  IR - controll commands for user interaction 
   *  see e_userCommands for indexes 
   */
  uint16_t irCommandCodes[UI_COMMAND_COUNT];

  /** color presets 
   *  5bit rgb  0bxbbbbbgggggrrrrr
   */
  uint16_t colorPresets[UI_COLOR_PRESET_COUNT];
}UserEepromParams;

#define USEREEPROMPARAMS_DEFAULT  {                          \
  /* .irAddress      = */ USER_ADDRESS_DEFAULT,              \
  /* .irCommandCodes = */ USER_COMMANDCODES_DEFAULTS,        \
  /* .colorPresets   = */ USER_COLORPRESETS_DEFAULTS         \
}
  


/** the delay after a keypress before another will be recognized */
#define USER_KEY_PRESS_DELAY_100ms   3

/** the delay between hue changes in auto-hue mode */
#define USER_HUE_CHANGE_INT_100ms   1
/** the delay between demo mode steps */
#define USER_DEMO_CHANGE_INT_100ms   5
/** how long the color profile number will be displayed */
#define USER_NORMAL_SHOW_NUMBER_DELAY_100ms 10

#define USER_STARTUP_WAIT_4_IR_TRAIN_s    7
#define USER_DELAY_BEFORE_SAVE_EEPROM_s 120

extern void    handle_ir_code ();


/**
 * Sets a new Time
 * @details  Encapsulates the display-routine setNewTime
 *           to realize diffrent modes of operation, which
 *           require the display not to be changed externally.
 *           So a call to this may not change the display.
 * @param i_time  the new time to set
 */
extern void  user_setNewTime( const DATETIME* i_time);

/**
 * does an animation step in different modes
 * Has to be called with 10Hz.
 */
extern void  user_isr10Hz(void);

/**
 * handles some long-term jobs
 * Has to be called with 1Hz.
 */
extern void  user_isr1Hz(void);

/** initializes the user module
 *  have to be called after initialization of eeprom
 */
extern void user_init(void);


#ifdef __cplusplus
}
#endif

#endif /* _WC_USER_H_ */
