/* ----------------------------------------------------------------------------
 *         ATMEL Microcontroller Software Support  -  ROUSSET  -
 * ----------------------------------------------------------------------------
 * Copyright (c) 2006, Atmel Corporation

 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaiimer below.
 *
 * - Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the disclaimer below in the documentation and/or
 * other materials provided with the distribution.
 *
 * Atmel's name may not be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * DISCLAIMER: THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * ----------------------------------------------------------------------------
 */

/*
 $Id: serial_example.c 107 2006-10-16 08:28:50Z jjoannic $

 This file has been modified by Thorsten Wilmer 2008 as a starting
 point for the USB-Display project.

 */

//------------------------------------------------------------------------------
//      Includes
//------------------------------------------------------------------------------

#include <ch.h>
#include "common.h"
#include "device.h"
#include "board.h"
#include "trace.h"
#include "usb.h"
#include "standard.h"
#include "cdc.h"
#include "serial_driver.h"
#include "lcd.h"

//------------------------------------------------------------------------------
//      Definitions
//------------------------------------------------------------------------------

//! \brief  Length of the loopback data buffer
#define USB_BUFFER_SIZE             50

//------------------------------------------------------------------------------
//      Prototypes
//------------------------------------------------------------------------------

//! \brief  Initialization callback
static void CBK_Init(const S_usb *pUsb);

//! \brief  Suspend callback
static void CBK_Suspend(const S_usb *pUsb);

//! \brief  Resume callback
static void CBK_Resume(const S_usb *pUsb);

//! \brief  New request callback
static void CBK_NewRequest(const S_usb *pUsb);

//! \brief  New reset callback
//static void CBK_Reset(const S_usb *pUsb);

//! \brief  New SOF callback
//static void CBK_SOF(const S_usb *pUsb);

//------------------------------------------------------------------------------
//      Internal variables
//------------------------------------------------------------------------------

//! \brief  List of endpoints (including endpoint 0) used by the device.
//! \see    S_usb_endpoint
static S_usb_endpoint pEndpoints[] = {

USB_ENDPOINT_SINGLEBANK, // Control endpoint 0
		USB_ENDPOINT_DUALBANK, // Data out endpoint
		USB_ENDPOINT_DUALBANK, // Data in endpoint
		USB_ENDPOINT_SINGLEBANK, // Notification endpoint
		};

//! \brief  Variable used to store the last received SETUP packet.
//! \see    S_usb_request
//! \see    S_usb
static S_usb_request sSetup;

//! \brief  Variable used to store the current device state
//! \see    S_usb
static unsigned int dState;

//! \brief  List of implemented callbacks
//! \see    S_usb_callbacks
//! \see    S_usb
static const S_usb_callbacks sCallbacks = {

CBK_Init, 0, // CBK_Reset
		CBK_Suspend, CBK_Resume, CBK_NewRequest, 0 // CBK_SOF
		};

//! \brief  USB driver instance
//! \see    S_usb
static const S_usb sUsb = {

&sDefaultDriver, pEndpoints, SER_NUM_ENDPOINTS, &sCallbacks, &sSetup, &dState };

//! \brief  CDC serial class driver instance
//! \see    S_ser
static S_ser sSer;

// Buffer for receiving data from the USB
static unsigned char pUsbBuffer[USB_BUFFER_SIZE];

//------------------------------------------------------------------------------
//      Internal Functions
//------------------------------------------------------------------------------

// Interrupt Service Routines
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
//! \brief  Spurious interrupt handler - do nothing, but necessary!
//
//------------------------------------------------------------------------------
void INTERRUPT     ISR_Spurious(void) {
	END_INTERRUPT();
}

//------------------------------------------------------------------------------
//! \brief  Handler for the USB controller interrupt
//!
//!         Defers the call to the USB_Handler function.
//------------------------------------------------------------------------------
void INTERRUPT     ISR_Driver(void) {
	USB_Handler(&sUsb);

	END_INTERRUPT();
}


// Callbacks
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//! \brief  Callback invoked during the initialization of the USB driver
//!
//!         Configures and enables USB controller and VBus monitoring interrupts
//! \param  pUsb    Pointer to a S_usb instance
//------------------------------------------------------------------------------
static void CBK_Init(const S_usb *pUsb) {
	// Configure spurious interrupt handler This is important! See this thread:
	// http://www.at91.com/www/phpBB2_mirror/viewtopic.php4?t=357
	*AT91C_AIC_SPU = (unsigned int) ISR_Spurious;

	// Configure and enable the USB controller interrupt
	AT91F_AIC_ConfigureIt(AT91C_BASE_AIC, USB_GetDriverID(pUsb),
			AT91C_AIC_PRIOR_LOWEST, 0, //AT91C_AIC_SRCTYPE_INT_HIGH_LEVEL,
			ISR_Driver);

	AT91F_AIC_EnableIt(AT91C_BASE_AIC, USB_GetDriverID(pUsb));

	// Power up the USB controller
	USB_Attach(pUsb);
	TRACE_INFO("CBK_Init USB-Attach\n\r");

}
//------------------------------------------------------------------------------
//! \brief  Callback invoked when the device becomes suspended
//!
//!         Disables LEDs (if they are used) and then puts the device into
//!         low-power mode. When traces are used, the device does not enter
//!         low-power mode to avoid losing some outputs.
//! \param  pUsb    Pointer to a S_usb instance
//------------------------------------------------------------------------------
static void CBK_Suspend(const S_usb *pUsb) {


#if defined(NOTRACES)
	DEV_Suspend();
#endif
}

//------------------------------------------------------------------------------
//! \brief  Callback invoked when the device leaves the suspended state
//!
//!         The device is first returned to a normal operating mode and LEDs are
//!         re-enabled. When traces are used, the device does not enter
//!         low-power mode to avoid losing some outputs.
//! \param  pUsb    Pointer to a S_usb instance
//------------------------------------------------------------------------------
static void CBK_Resume(const S_usb *pUsb) {
#if defined(NOTRACES)
	DEV_Resume();
#endif

	TRACE_INFO("CBK_Resume\n\r");

}

//------------------------------------------------------------------------------
//! \brief  Callback invoked when a new SETUP request is received
//!
//!         The new request if forwarded to the standard request handler,
//!         which performs the enumeration of the device.
//! \param  pUsb   Pointer to a S_usb instance
//------------------------------------------------------------------------------
static void CBK_NewRequest(const S_usb *pUsb) {
	SER_RequestHandler(&sSer);
}




// Other functions
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//! \brief  processes data receiving from the USB host
//!
//!         This function operates asynchronously.
//!         It shall be modified to queue the incoming Data
//! \param  pBuffer           Must be 0 when function is first called
//! \param  bStatus           Same as above
//! \param  dBytesTransferred Same as above
//! \param  dBytesRemaining   Same as above
//------------------------------------------------------------------------------


static void ReadDataFormUSB(unsigned char *pBuffer, unsigned char bStatus,
		unsigned int dBytesTransferred, unsigned int dBytesRemaining) {
	if (pBuffer != 0) {

		if (bStatus != SER_STATUS_SUCCESS) {

			TRACE_WARNING("W: ForwardUSB2Serial: Transfer error\n\r");
		} else {
			/* Hier kommen die Nachrichten vom Treiber rein
			 In zukunft sollte hier ein Event ans OS geschickt werden,
			 die Daten in eine Queue gesteckt werden.
			 Außerdem sollten alle Daten abgenommmen und
			 anhand der mit übertragenen Länge einzel Befehle
			 erkannt werden. Aber für einen Ersten Wurf reicht dies aus*/

			if (dBytesTransferred > 21)
				dBytesTransferred = 21;
			pBuffer[dBytesTransferred] = 0;
			TRACE_DEBUG_M("%d bytes received, %d bytes remaining, forwarding 0x%x\n\r",
					dBytesTransferred, dBytesRemaining, pBuffer[0]);

			TRACE_INFO("Data Received: %s\n", pBuffer);
			print(pBuffer);

		}
	}

	do {

		bStatus = SER_Read(&sSer, pUsbBuffer, USB_BUFFER_SIZE,
				(Callback_f) ReadDataFormUSB, pUsbBuffer);
	} while (bStatus != SER_STATUS_SUCCESS);
}



static WorkingArea(waThread1, 64);
static Mutex exConsole;
void doPrint(char *c) {
	chMtxLock(&exConsole);
	print(c);
	chMtxUnlock();
}

static msg_t Thread1(void *arg) {

	while (TRUE) {
		chThdSleep(1000);

	}
	return 0;
}

static WorkingArea(waThread2, 64);
static msg_t Thread2(void *arg) {

	while (TRUE) {

		chThdSleep(1000);
	}
	return 0;
}

// "reset reason" handling
extern unsigned long reset_reason;
extern unsigned long link_register;
extern unsigned long abort_link_register;
const char *reset_reasons[] = { "normal", "undef", "swi", "preab", "datab" };
const char *at91_reasons[] = { "power-up", "1", "watchdog", "software", "user",
		"brownout", "6", "7" };

void trace_reset_reason(void) {
	TRACE_INFO("Reset reasons %d/%d: %s / %s\r\n",
			reset_reason, ((*AT91C_RSTC_RSR)>>8) & 7,
			reset_reasons[reset_reason], at91_reasons[((*AT91C_RSTC_RSR)>>8) & 7]);
	TRACE_INFO("Abort status %X, address %X, link %X, abort link %X\r\n",
			*AT91C_MC_ASR, *AT91C_MC_AASR, link_register, abort_link_register);
}
extern void hwinit();
//------------------------------------------------------------------------------
//          Main
//------------------------------------------------------------------------------
int main() {

	*AT91C_AIC_SPU = (unsigned int) ISR_Spurious;
	AT91F_RSTSetMode(AT91C_BASE_RSTC, AT91C_RSTC_URSTEN);
	chSysInit();
	TRACE_INIT();
	USB_Disconnect(&sUsb);
	SER_Init(&sSer, &sUsb);
	ReadDataFormUSB(0, 0, 0, 0); // Initialize Data reception

	chMtxInit(&exConsole);
	// So würde man einen Thread starten, ja die werden im Moment nicht benutzt....
	//chThdCreate(NORMALPRIO, 0, waThread1, sizeof(waThread1), Thread1, NULL);
	//chThdCreate(NORMALPRIO, 0, waThread2, sizeof(waThread2), Thread2, NULL);


	trace_reset_reason();

	unsigned char bStatus;

	initialize_lcd();
	while (1) {

		updateData();

		printDisplayData();

		chThdSleep(10);
	}


}

