/*
    ChibiOS/RT - Copyright (C) 2006-2007 Giovanni Di Sirio.

    This file is part of ChibiOS/RT.

    ChibiOS/RT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    ChibiOS/RT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

    Modified by Thorsten Wilmer 2008 to suite the needs for the
     usb-display project
*/

/*
 * Generic ARM startup file for ChibiOS/RT (Atmel variant).
 */

.set    MODE_USR, 0x10
.set    MODE_FIQ, 0x11
.set    MODE_IRQ, 0x12
.set    MODE_SVC, 0x13
.set    MODE_ABT, 0x17
.set    MODE_UND, 0x1B
.set    MODE_SYS, 0x1F

.equ    I_BIT, 0x80
.equ    F_BIT, 0x40
   .equ    RST_BASE, 0xFFFFFD00    // Reset Control Base
        .equ    RST_RC, 0x00            // Reset control register

   .equ    WDT_BASE, 0xFFFFFD40    // watch dog timer base address
   .equ    WDT_MR, 0x04            // WDT mode register
   .equ    MC_BASE, 0xFFFFFF00     // memory controller base address
   .equ    MC_FMR, 0x60            // offset of flash mode reg

      .equ    PMC_BASE, 0xFFFFFC00    // power management controller base address
        .equ    PMC_SR, 0x68            // status register
        .equ    PMC_MOR, 0x20           // main oscillator register
        .equ    PMC_PLLR, 0x2C          // PLL clock register
        .equ    PMC_MCKR, 0x30          // master clock register





.text
.code 32
.balign 4
/*
 * System entry points.
 */
_start:
        ldr     pc, _reset
        ldr     pc, _undefined
        ldr     pc, _swi
        ldr     pc, _prefetch
        ldr     pc, _abort
        nop
        ldr     pc, [pc,#-0xF20]        /* AIC - AIC_IVR */
        ldr     pc, [pc,#-0xF20]        /* AIC - AIC_FVR */

_reset:
        .word   ResetHandler
b ResetHandler
_undefined:
        .word   UndHandler
b _undefined
_swi:
        .word   SwiHandler
b _swi
_prefetch:
        .word   PrefetchHandler
b _prefetch
_abort:
        .word   AbortHandler

_fiq:
        .word   FiqHandler
        .word   0
        .word   0

/*
 * Reset handler.
 */
ResetHandler:

 // note: you can only touch the WDT_MR once (so should let main code do what it wants)
        // turn off watchdog timer
        ldr     r0, =WDT_BASE
        ldr     r1, =0x00008000         // set WDDIS bit (watchdog disable)
        str     r1, [r0, #WDT_MR]

        ldr     r0, =RST_BASE
        ldr     r1, =0xa5000003         // set WDDIS bit (watchdog disable)
        str     r1, [r0, #WDT_MR]
        // setup embedded flash controller (EFC)
        // (we need to do this before setting a high frequency MCK, otherwise
        // flash reads won't use the wait state they need)
        ldr     r0, =MC_BASE
        ldr     r1, =0x00480100         // 72 (0x48) master clocks in 1.5 us
        str     r1, [r0, #MC_FMR]       // need 1 wait state with 48 MHz MCK

        // enable main oscillator (18.432 MHz) and wait till stabilised
        ldr     r0, =PMC_BASE
        ldr     r1, =0x00001001         // wait 16*8=128 RC clocks
        str     r1, [r0, #PMC_MOR]      // ~3 ms (if RC worst case 42 kHz)
oscillate:
        ldr     r2, [r0, #PMC_SR]       // read status register
        ands    r2, r2, #1              // main oscillator stable?
        beq     oscillate               // no, keep waiting

        // Enable PLL clock and wait till stabilised. We need a 48 MHz clock for
        // USB, but the minimum PLL frequency is 80 MHz, so go for 48*2=96 MHz.
        // 96 MHz means 250 for the multiplier (value actually 250-1), and      48
        // for the divider, because 18.432*250/48 = 96 MHz. We also have to set
        // the PLL output frequency range to the 80-160 MHz range.
        ldr     r1, =0x10F92030         // USB clock is PLL/2, mul=249+1
        str     r1, [r0, #PMC_PLLR]     // out=80-160MHz, counter=32, div=48
pllate:
        ldr     r2, [r0, #PMC_SR]       // read status register
        ands    r2, r2, #4              // PLL locked and stable?
        beq     pllate                  // no, keep waiting

        // select master clock as PLL/2 (48 MHz)
        ldr     r1, =7
        str     r1, [r0, #PMC_MCKR]



        /*
         * Stack pointers initialization.
         */
        ldr     r0, =__ram_end__
        /* Undefined */
        msr     CPSR_c, #MODE_UND | I_BIT | F_BIT
        mov     sp, r0
        ldr     r1, =__und_stack_size__
        sub     r0, r0, r1
        /* Abort */
        msr     CPSR_c, #MODE_ABT | I_BIT | F_BIT
        mov     sp, r0
  mov     r8, lr                  // save abort link register in r8 for later
        ldr     r1, =__abt_stack_size__
        sub     r0, r0, r1
        /* FIQ */
        msr     CPSR_c, #MODE_FIQ | I_BIT | F_BIT
        mov     sp, r0
        ldr     r1, =__fiq_stack_size__
        sub     r0, r0, r1
        /* IRQ */
        msr     CPSR_c, #MODE_IRQ | I_BIT | F_BIT
        mov     sp, r0
        ldr     r1, =__irq_stack_size__
        sub     r0, r0, r1
        /* Supervisor */
        msr     CPSR_c, #MODE_SVC | I_BIT | F_BIT
        mov     sp, r0
        ldr     r1, =__svc_stack_size__
        sub     r0, r0, r1
        /* System */
        msr     CPSR_c, #MODE_SYS | I_BIT | F_BIT
        mov     sp, r0
        ldr     r1, =__sys_stack_size__
        sub     r0, r0, r1

        // *after* copying data, save low-level reset reason and link register
        ldr     r0, =reset_reason
        str     r7, [r0, #0]
        ldr     r0, =link_register
        str     lr, [r0, #0]
        ldr     r0, =abort_link_register
        str     r8, [r0, #0]

        /*
         * Data initialization.
         * NOTE: It assumes that the DATA size is a multiple of 4.
         */
        ldr     r1, =_textdata
        ldr     r2, =_data
        ldr     r3, =_edata
dataloop:
        cmp     r2, r3
        ldrlo   r0, [r1], #4
        strlo   r0, [r2], #4
        blo     dataloop
        /*
         * BSS initialization.
         * NOTE: It assumes that the BSS size is a multiple of 4.
         */
        mov     r0, #0
        ldr     r1, =_bss_start
        ldr     r2, =_bss_end
bssloop:
        cmp     r1, r2
        strlo   r0, [r1], #4
        blo     bssloop
        /*
         * Application-provided HW initialization routine.
         */
mrs r0, cpsr
bic r0,r0,#0x80
msr cpsr_c,r0

#ifndef THUMB_NO_INTERWORKING
        bl      hwinit
        /*
         * main(0, NULL).
         */
        mov     r0, #0
        mov     r1, r0
        bl      main
        bl      chSysHalt
#else
        add     r0, pc, #1
        bx      r0
.code 16
        bl      hwinit
        mov     r0, #0
        mov     r1, r0
        bl      main
        bl      chSysHalt
.code 32
#endif

.globl _loop
_loop: b        _loop
.weak UndHandler
.globl UndHandler
UndHandler:
    mov     r7, #5
    b _loop

.weak SwiHandler
.globl SwiHandler
SwiHandler:
    mov     r7, #4
    b _loop

.weak PrefetchHandler
.globl PrefetchHandler
PrefetchHandler:
    mov     r7, #3
    b _loop

.weak AbortHandler
.globl AbortHandler
AbortHandler:
    mov     r7, #2
        ldr     r0, =MC_BASE
        ldr     r1, [r0, #0]
        ldr     r2, [r0, #4]
        ldr     r3, [r0, #8]
        ldr     r4, =0x42424242
        ldr     r9, =0x42424242
        ldr     r10, [r0, #0]
        ldr     r12, [r0, #4]
        ldr     r8, [r0, #8]

loop_abort:
b _abort

    b _loop

.weak FiqHandler
.globl FiqHandler
FiqHandler:
    mov     r7, #1
    b _loop

.global default_irq_handler
default_irq_handler:
    b default_irq_handler


        .global reset_reason
        .data
        .align  4
reset_reason:
        .word   0

        .global link_register
link_register:
        .word   0

        .global abort_link_register
abort_link_register:
        .word   0

