/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * @file mcurses.h - include file for mcurses lib
 *
 * Copyright (c) 2011 Frank Meyer - frank(at)fli4l.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

#include <stdint.h>

#ifndef unix
#include <avr/pgmspace.h>
#endif

#include "mcurses-config.h"

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * Attributes, may be ORed
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#define A_NORMAL                0x00                                                // normal
#define A_UNDERLINE             0x01                                                // underline
#define A_REVERSE               0x02                                                // reverse
#define A_BLINK                 0x04                                                // blink
#define A_BOLD                  0x08                                                // bold
#define A_DIM                   0x10                                                // dim
#define A_STANDOUT              A_BOLD                                              // standout (same as bold)

extern void                     initscr (void);                                     // initialize mcurses

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * mcurses functions
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
extern void                     move (uint8_t, uint8_t);                            // move cursor to line, column (home = 0, 0)
extern void                     attrset (uint8_t);                                  // set attribute(s)
extern void                     addch (uint8_t);                                    // add a character
extern void                     addstr (char *);                                    // add a string
extern void                     addstr_P (const char *);                            // add a string (PROGMEM)
extern void                     setscrreg (uint8_t, uint8_t);                       // set scrolling region
extern void                     deleteln (void);                                    // delete line at current line position
extern void                     insertln (void);                                    // insert a line at current line position
extern void                     scroll (void);                                      // scroll line up
extern void                     clear (void);                                       // clear total screen
extern void                     clrtobot (void);                                    // clear screen from current line to bottom
extern void                     clrtoeol (void);                                    // clear from current column to end of line
extern void                     delch (void);                                       // delete character at current position
extern void                     insch (uint8_t);                                    // insert character at current position
extern uint8_t                  getch (void);                                       // read key

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * mcurses macros
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#define erase()                 clear()                                             // clear total screen, same as clear()
#define mvaddch(y,x,c)          move((y),(x)), addch((c))                           // move cursor, then add character
#define mvaddstr(y,x,s)         move((y),(x)), addstr((s))                          // move cursor, then add string
#define mvaddstr_P(y,x,s)       move((y),(x)), addstr_P((s))                        // move cursor, then add string (PROGMEM)
#define mvinsch(y,x,c)          move((y),(x)), insch((c))                           // move cursor, then insert character
#define mvdelch(y,x)            move((y),(x)), delch()                              // move cursor, then delete character
#ifdef unix
#define refresh()               fflush (stdout);                                    // this is a dummy yet
#else
#define refresh()                                                                   // this is a dummy yet
#endif
#define endwin()                move (LINES - 1, 0)                                 // end mcurses

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * mcurses keys
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#define KEY_ESCAPE              0x1B                                                // ESCAPE (pressed twice)

#define KEY_DOWN                0x80                                                // Down arrow key
#define KEY_UP                  0x81                                                // Up arrow key
#define KEY_LEFT                0x82                                                // Left arrow key
#define KEY_RIGHT               0x83                                                // Right arrow key
#define KEY_HOME                0x84                                                // Home key
#define KEY_DC                  0x85                                                // Delete character key
#define KEY_IC                  0x86                                                // Ins char/toggle ins mode key
#define KEY_NPAGE               0x87                                                // Next-page key
#define KEY_PPAGE               0x88                                                // Previous-page key
#define KEY_END                 0x89                                                // End key
#define KEY_BTAB                0x8A                                                // Back tab key
#define KEY_F1                  0x8B                                                // Function key F1
#define KEY_F(n)                (KEY_F1+(n)-1)                                      // Space for add. 12 function keys
