/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * @file mcurses.c - mcurses lib
 *
 * Copyright (c) 2011 Frank Meyer - frank(at)fli4l.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef unix
#include <termio.h>
#define PROGMEM
#define PSTR(x)                                 (x)
#define pgm_read_byte(s)                        (*s)
#else
#include <avr/io.h>
#include <avr/pgmspace.h>
#endif

#include "mcurses.h"

static uint8_t                                  mcurses_attrs;
static uint8_t                                  mcurses_scrl_start = 0;
static uint8_t                                  mcurses_scrl_end = LINES - 1;
static uint8_t                                  mcurses_cury;
static uint8_t                                  mcurses_curx;

#ifdef unix
struct termio                                   _oldmode, _newmode;
#else

#include <util/setbaud.h>

#ifdef UBRR0H

#define UART0_UBRRH                             UBRR0H
#define UART0_UBRRL                             UBRR0L
#define UART0_UCSRA                             UCSR0A
#define UART0_UCSRB                             UCSR0B
#define UART0_UCSRC                             UCSR0C
#define UART0_UDRE_BIT_VALUE                    (1<<UDRE0)
#define UART0_UCSZ1_BIT_VALUE                   (1<<UCSZ01)
#define UART0_UCSZ0_BIT_VALUE                   (1<<UCSZ00)
#ifdef URSEL0
#define UART0_URSEL_BIT_VALUE                   (1<<URSEL0)
#else
#define UART0_URSEL_BIT_VALUE                   (0)
#endif
#define UART0_TXEN_BIT_VALUE                    (1<<TXEN0)
#define UART0_RXEN_BIT_VALUE                    (1<<RXEN0)
#define UART0_UDR                               UDR0
#define UART0_U2X                               U2X0
#define UART0_RXC                               RXC0

#else

#define UART0_UBRRH                             UBRRH
#define UART0_UBRRL                             UBRRL
#define UART0_UCSRA                             UCSRA
#define UART0_UCSRB                             UCSRB
#define UART0_UCSRC                             UCSRC
#define UART0_UDRE_BIT_VALUE                    (1<<UDRE)
#define UART0_UCSZ1_BIT_VALUE                   (1<<UCSZ1)
#define UART0_UCSZ0_BIT_VALUE                   (1<<UCSZ0)
#ifdef URSEL
#define UART0_URSEL_BIT_VALUE                   (1<<URSEL)
#else
#define UART0_URSEL_BIT_VALUE                   (0)
#endif
#define UART0_TXEN_BIT_VALUE                    (1<<TXEN)
#define UART0_RXEN_BIT_VALUE                    (1<<RXEN)
#define UART0_UDR                               UDR
#define UART0_U2X                               U2X
#define UART0_RXC                               RXC

#endif

#endif // ! unix

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * UART: init
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void
mcurses_uart_init (void)
{
#ifndef unix
    UART0_UBRRH = UBRRH_VALUE;                                                                      // set baud rate
    UART0_UBRRL = UBRRL_VALUE;

#if USE_2X
    UART0_UCSRA |= (1<<UART0_U2X);
#else
    UART0_UCSRA &= ~(1<<UART0_U2X);
#endif

    UART0_UCSRC = UART0_UCSZ1_BIT_VALUE | UART0_UCSZ0_BIT_VALUE | UART0_URSEL_BIT_VALUE;
    UART0_UCSRB |= UART0_TXEN_BIT_VALUE | UART0_RXEN_BIT_VALUE;                                     // enable UART TX
#endif // ! unix
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * UART: send character
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void
mcurses_uart_putc (uint8_t ch)
{
#ifdef unix
    putchar (ch);
#else
    while (!(UART0_UCSRA & UART0_UDRE_BIT_VALUE))
    {
        ;
    }

    UART0_UDR = ch;
#endif // ! unix
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * UART: receive character with wait
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
uint8_t
mcurses_uart_getc (void)
{
    uint8_t ch;

#ifdef unix

    ch = getchar ();

#else

    while (!(UART0_UCSRA & (1<<UART0_RXC)))             // character available?
    {                                                   // no
        ;                                               // wait
    }

    ch = UART0_UDR;                                     // read character from UDRx

#endif // ! unix

    return (ch);
}


/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * INTERN: convert number into string
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#ifdef unix
static char *
myitoa (int x, char * buf)
{
    sprintf (buf, "%d", x);
    return (buf);
}
#else
#define myitoa(x,buf)   itoa ((x), buf, 10)
#endif

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * INTERN: put a string (raw)
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void
mcurses_puts (char * str)
{
    while (*str)
    {
        mcurses_uart_putc (*str++);
    }
}

static void
mcurses_puts_P (const char * str)
{
    uint8_t ch;

    while ((ch = pgm_read_byte(str)) != '\0')
    {
        mcurses_uart_putc (ch);
        str++;
    }
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * INTERN: set scrolling region (raw)
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void
mysetscrreg (uint8_t top, uint8_t bottom)
{
    char buf[4];

    if (top == bottom)
    {
        mcurses_puts_P (PSTR("\033[r"));                                    // reset scrolling region
    }
    else
    {
        mcurses_puts_P (PSTR("\033["));
        mcurses_puts (myitoa (top + 1, buf));
        mcurses_uart_putc (';');
        mcurses_puts (myitoa (bottom + 1, buf));
        mcurses_uart_putc ('r');
    }
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * move cursor (raw)
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void
mymove (uint8_t y, uint8_t x)
{
    char buf[4];

    mcurses_puts_P (PSTR("\033["));
    mcurses_puts (myitoa (y + 1, buf));
    mcurses_uart_putc (';');
    mcurses_puts (myitoa (x + 1, buf));
    mcurses_uart_putc ('H');
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: initialize
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
initscr (void)
{
    mcurses_uart_init ();
#ifdef unix
    (void) ioctl (0, TCGETA, &_oldmode);
    (void) ioctl (0, TCGETA, &_newmode);

    _newmode.c_lflag &= ~ICANON;
    _newmode.c_lflag &= ~ECHO;
    _newmode.c_oflag &= ~TAB3;                                                  // switch off TAB conversion
    _newmode.c_cc[VINTR] = '\377';                                              // disable VINTR VQUIT
    _newmode.c_cc[VQUIT] = '\377';                                              // but don't touch VSWTCH
    _newmode.c_cc[VMIN] = 1;                                                    // block input:
    _newmode.c_cc[VTIME] = 0;                                                   // one character
    (void) ioctl (0, TCSETAW, &_newmode);

#endif

    clear ();
    mcurses_attrs = A_NORMAL;
    move (0, 0);
}

#ifdef unix
void
endwin (void)
{
    move (LINES - 1, 0);
    refresh ();
    (void) ioctl (0, TCSETAW, &_oldmode);
}
#endif

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: add character
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
addch (uint8_t ch)
{
    mcurses_uart_putc (ch);
    mcurses_curx++;
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: add string
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
addstr (char * str)
{
    while (*str)
    {
        mcurses_uart_putc (*str++);
    }
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: add string
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
addstr_P (const char * str)
{
    uint8_t ch;

    while ((ch = pgm_read_byte(str)) != '\0')
    {
        mcurses_uart_putc (ch);
        str++;
    }
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: set attribute(s)
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
attrset (uint8_t attr)
{
    mcurses_puts_P (PSTR("\033[0"));

    if (attr & A_REVERSE)
    {
        mcurses_puts_P (PSTR(";7"));
    }
    if (attr & A_UNDERLINE)
    {
        mcurses_puts_P (PSTR(";4"));
    }
    if (attr & A_BLINK)
    {
        mcurses_puts_P (PSTR(";5"));
    }
    if (attr & A_BOLD)
    {
        mcurses_puts_P (PSTR(";1"));
    }
    if (attr & A_DIM)
    {
        mcurses_puts_P (PSTR(";2"));
    }
    mcurses_uart_putc ('m');
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: move cursor
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
move (uint8_t y, uint8_t x)
{
    mcurses_cury = y;
    mcurses_curx = x;
    mymove (y, x);
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: delete line
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
deleteln (void)
{
    mysetscrreg (mcurses_scrl_start, mcurses_scrl_end);             // set scrolling region
    mymove (mcurses_cury, 0);                                       // goto to current line
    mcurses_puts_P (PSTR("\033[M"));                                // delete line
    mysetscrreg (0, 0);                                             // reset scrolling region
    move (mcurses_cury, mcurses_curx);                              // restore position
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: insert line
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
insertln (void)
{
    mysetscrreg (mcurses_cury, mcurses_scrl_end);                   // set scrolling region
    mymove (mcurses_cury, 0);                                       // goto to current line
    mcurses_puts_P (PSTR("\033[L"));                                // insert line
    mysetscrreg (0, 0);                                             // reset scrolling region
    mymove (mcurses_cury, mcurses_curx);                            // restore position
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: scroll
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
scroll (void)
{
    mysetscrreg (mcurses_scrl_start, mcurses_scrl_end);             // set scrolling region
    mymove (mcurses_scrl_end, 0);                                   // goto to last line of scrolling region
    mcurses_puts_P (PSTR("\033E"));                                 // next line
    mysetscrreg (0, 0);                                             // reset scrolling region
    mymove (mcurses_cury, mcurses_curx);                            // restore position
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: clear
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
clear (void)
{
    mcurses_puts_P (PSTR("\033[2J"));
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: clear to bottom of screen
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
clrtobot (void)
{
    mcurses_puts_P (PSTR("\033[J"));
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: clear to end of line
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
clrtoeol (void)
{
    mcurses_puts_P (PSTR("\033[K"));
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: delete character at cursor position
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
delch (void)
{
    mcurses_puts_P (PSTR("\033[P"));
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: insert character
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
insch (uint8_t ch)
{
    mcurses_puts_P (PSTR("\033[4h"));
    mcurses_uart_putc (ch);
    mcurses_curx++;
    mcurses_puts_P (PSTR("\033[4l"));
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: set scrolling region
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
setscrreg (uint8_t t, uint8_t b)
{
    mcurses_scrl_start = t;
    mcurses_scrl_end = b;
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * MCURSES: read key
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#define MAX_KEYS                ((KEY_F1 + 12) - 0x80)

static const char * function_keys[MAX_KEYS] =
{
    "B",                        // KEY_DOWN                 0x80                                // Down arrow key
    "A",                        // KEY_UP                   0x81                                // Up arrow key
    "D",                        // KEY_LEFT                 0x82                                // Left arrow key
    "C",                        // KEY_RIGHT                0x83                                // Right arrow key
    "2~",                       // KEY_HOME                 0x84                                // Home key
    "4~",                       // KEY_DC                   0x85                                // Delete character key
    "1~",                       // KEY_IC                   0x86                                // Ins char/toggle ins mode key
    "6~",                       // KEY_NPAGE                0x87                                // Next-page key
    "3~",                       // KEY_PPAGE                0x88                                // Previous-page key
    "5~",                       // KEY_END                  0x89                                // End key
    "Z",                        // KEY_BTAB                 0x8A                                // Back tab key
    "11~",                      // KEY_F(1)                 0x8B
    "12~",                      // KEY_F(2)                 0x8C
    "13~",                      // KEY_F(3)                 0x8D
    "14~",                      // KEY_F(4)                 0x8E
    "15~",                      // KEY_F(5)                 0x8F
    "17~",                      // KEY_F(6)                 0x90
    "18~",                      // KEY_F(7)                 0x91
    "19~",                      // KEY_F(8)                 0x92
    "20~",                      // KEY_F(9)                 0x93
    "21~",                      // KEY_F(10)                0x94
    "23~",                      // KEY_F(11)                0x95
    "24~"                       // KEY_F(12)                0x96
};

uint8_t
getch (void)
{
    char    buf[4];
    uint8_t ch;
    uint8_t idx;

    ch = mcurses_uart_getc ();

    if (ch == '\033')                                                                                   // ESCAPE
    {
        ch = mcurses_uart_getc ();

        if (ch == '\033')                                                                                   // ESCAPE
        {
            return KEY_ESCAPE;
        }
        else if (ch == '[')
        {
            for (idx = 0; idx < 3; idx++)
            {
                ch = mcurses_uart_getc ();

                buf[idx] = ch;

                if ((ch >= 'A' && ch <= 'Z') || ch == '~')
                {
                    idx++;
                    break;
                }
            }

            buf[idx] = '\0';

            for (idx = 0; idx < MAX_KEYS; idx++)
            {
                if (! strcmp (buf, function_keys[idx]))
                {
                    ch = idx + 0x80;
                    break;
                }
            }

            if (idx == MAX_KEYS)
            {
                ch = KEY_ESCAPE;
            }
        }
    }

    return ch;
}
