/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * @file demo.c - demo program using mcurses lib
 *
 * Copyright (c) 2011 Frank Meyer - frank(at)fli4l.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>

#ifdef unix
#include <unistd.h>
#else
#include <avr/io.h>
#include <util/delay.h>
#endif

#include "mcurses.h"

#ifdef unix
#define         PAUSE(x)                        { refresh(); if (!fast) usleep (1000 * x); }
static char *   myitoa(uint8_t x, char * buf)   { sprintf (buf, "%d", x); return buf; }
#define         PSTR(x)                         (x)
#else
#define         PAUSE(x)                        { refresh(); if (!fast) _delay_ms (x); }
#define         myitoa(x,buf)                   itoa ((x), buf, 10)
#endif

static uint8_t  fast;

static void
show_top_line_P (const char * str)
{
    int     col;

    move (1, 0);
    attrset (A_REVERSE);

    for (col = 0; col < COLS; col++)
    {
        addch (' ');
    }

    mvaddstr_P (1, 2, str);
    attrset (A_NORMAL);
}

static void
show_bottom_line_P (const char * str)
{
    uint8_t col;

    move (LINES - 3, 0);
    attrset (A_REVERSE);

    for (col = 0; col < COLS; col++)
    {
        addch (' ');
    }

    mvaddstr_P (LINES - 3, 2, str);
    attrset (A_NORMAL);
}

static void
message_P (char * msg)
{
    move (LINES - 2, 0);
    addstr_P (msg);
    clrtoeol ();
}

static void
shift_left (uint8_t y, uint8_t x, uint8_t ch)
{
    uint8_t col;

    move (y, COLS - 2);
    addch (ch);
    move (y, x);

    for (col = COLS - 2; col > x; col--)
    {
        PAUSE (5);
        delch ();
    }
}

static void
shift_left_str (uint8_t y, uint8_t x, char * str)
{
    char *  s;
    uint8_t xx = x;

    for (s = str; *s; s++)
    {
        if (*s != ' ')
        {
            shift_left (y, xx, *s);
        }
        xx++;
    }

    move (y, x);
    attrset (A_REVERSE);

    for (s = str; *s; s++)
    {
        addch (*s);
        PAUSE (25);
    }

    move (y, x);
    attrset (A_NORMAL);

    for (s = str; *s; s++)
    {
        addch (*s);
        PAUSE (25);
    }
}

static void
screen_demo (void)
{
    char    buf[10];
    uint8_t line;
    uint8_t col;

    clear ();
    show_top_line_P (PSTR("TOP LINE 2"));
    show_bottom_line_P (PSTR("BOTTOM LINE 22"));
    setscrreg (2, LINES - 4);

    if (fast)
    {
        mvaddstr_P (10, 20, PSTR("MCURSES LIB DEMO IN FAST MOTION"));
    }
    else
    {
        shift_left_str (10, 20, "MCURSES LIB DEMO IN SLOW MOTION");
    }

    for (line = 0; line < 5; line++)
    {
        scroll ();
        PAUSE (200);
    }

    move (5, 15);
    for (line = 0; line < 5; line++)
    {
        insertln ();
        PAUSE (200);
    }

    move (10, 18);
    for (col = 0; col < 5; col ++)
    {
        insch (' ');
        PAUSE (200);
    }

    move (10, 18);
    for (col = 0; col < 5; col ++)
    {
        delch ();
        PAUSE (200);
    }

    clear ();

    show_top_line_P (PSTR("TOP LINE 2"));
    show_bottom_line_P (PSTR("BOTTOM LINE 22"));

    message_P (PSTR("line positioning test"));

    for (line = 2; line <= LINES - 4; line++)
    {
        move (line, 0);
        addstr (myitoa (line + 1, buf));
    }

    PAUSE (700);

    message_P (PSTR("BOLD attribute test"));
    attrset (A_BOLD);
    mvaddstr_P (10, 10, PSTR("BOLD"));
    attrset (A_NORMAL);
    PAUSE (700);

    message_P (PSTR("REVERSE attribute test"));
    attrset (A_REVERSE);
    mvaddstr_P (11, 10, PSTR("REVERSE"));
    attrset (A_NORMAL);
    PAUSE (700);

    message_P (PSTR("insert test"));
    for (col = 10; col <= 22; col += 2)
    {
        mvinsch (11, col, ' ');
    }
    move (11, col + 1);
    PAUSE (700);

    message_P (PSTR("UNDERLINE attribute test"));
    attrset (A_UNDERLINE);
    mvaddstr_P (12, 10, PSTR("UNDERLINE"));
    attrset (A_NORMAL);
    PAUSE (1000);

    message_P (PSTR("insert line test"));
    move (11, 10);
    insertln ();
    PAUSE (1000);

    addstr_P (PSTR("Inserted line, will be deleted soon..."));
    PAUSE (1000);

    message_P (PSTR("delete line test"));
    move (11, 10);
    deleteln ();
    PAUSE (1000);

    message_P (PSTR("scroll up line test"));
    for (line = 0; line < LINES - 4; line++)
    {
        scroll ();
        PAUSE (50);
    }
}

static void
temperature ()
{
    uint8_t idx;
    uint8_t t;
    uint8_t x;
    uint8_t loop;
    char    buf[10];
    char    temp[15] = { 0, 8, 15, 21, 26, 30, 32, 35, 32, 30, 26, 21, 15, 8, 0 };

    clear ();
    show_top_line_P (PSTR("Temperatures in a disk storage"));
    show_bottom_line_P (PSTR(""));

    for (loop = 0; loop < 255; loop++)
    {
        for (idx = 0; idx < 15; idx++)
        {
            mvaddstr (idx + 4, 5, "Disk ");
            myitoa (idx + 1, buf);

            if (idx + 1 < 10)
            {
                addch (' ');
            }

            addstr (buf);
            addstr (": ");
            myitoa (temp[idx] + 20, buf);
            addstr (buf);
            addch ('');

            move (idx + 4, 20);
            attrset (A_REVERSE);

            for (t = 0; t < temp[idx]; t++)
            {
                addch (' ');
            }

            attrset (A_NORMAL);
            clrtoeol ();

            x = rand() & 0x1F;

            if (x == 0x01)
            {
                if (temp[idx] < 55)
                {
                    temp[idx]++;
                }
            }
            else if (x == 0x02)
            {
                if (temp[idx] > 0)
                {
                    temp[idx]--;
                }
            }
        }
        PAUSE (100);
    }
}

int
main ()
{
    char    buf[10];
    uint8_t idx;
    uint8_t ch;

    initscr ();

    while (1)
    {
        screen_demo ();
        clear ();
        mvaddstr_P (10, 10, PSTR("Now the same in full speed... "));

        for (idx = 5; idx > 0; idx--)
        {
            move (10, 40);
            myitoa(idx, buf);
            addstr (buf);
            PAUSE (1000);
        }

        fast = 1;
        screen_demo ();
        clear ();
        fast = 0;

        mvaddstr_P (10, 10, PSTR("Whoops, too fast? Let's do something else... "));
        PAUSE (5000);

        temperature ();
        clear ();

        while (1)
        {
            move (10, 10);
            addstr_P (PSTR("Press a key (2x ESC to quit): "));

            ch = getch ();

            switch (ch)
            {
                case '\t':          addstr_P (PSTR("TAB"));         break;
                case '\r':          addstr_P (PSTR("CR"));          break;
                case KEY_ESCAPE:    addstr_P (PSTR("KEY_ESCAPE"));  break;
                case KEY_DOWN:      addstr_P (PSTR("KEY_DOWN"));    break;
                case KEY_UP:        addstr_P (PSTR("KEY_UP"));      break;
                case KEY_LEFT:      addstr_P (PSTR("KEY_LEFT"));    break;
                case KEY_RIGHT:     addstr_P (PSTR("KEY_RIGHT"));   break;
                case KEY_HOME:      addstr_P (PSTR("KEY_HOME"));    break;
                case KEY_DC:        addstr_P (PSTR("KEY_DC"));      break;
                case KEY_IC:        addstr_P (PSTR("KEY_IC"));      break;
                case KEY_NPAGE:     addstr_P (PSTR("KEY_NPAGE"));   break;
                case KEY_PPAGE:     addstr_P (PSTR("KEY_PPAGE"));   break;
                case KEY_END:       addstr_P (PSTR("KEY_END"));     break;
                case KEY_BTAB:      addstr_P (PSTR("KEY_BTAB"));    break;
                case KEY_F(1):      addstr_P (PSTR("KEY_F(1)"));    break;
                case KEY_F(2):      addstr_P (PSTR("KEY_F(2)"));    break;
                case KEY_F(3):      addstr_P (PSTR("KEY_F(3)"));    break;
                case KEY_F(4):      addstr_P (PSTR("KEY_F(4)"));    break;
                case KEY_F(5):      addstr_P (PSTR("KEY_F(5)"));    break;
                case KEY_F(6):      addstr_P (PSTR("KEY_F(6)"));    break;
                case KEY_F(7):      addstr_P (PSTR("KEY_F(7)"));    break;
                case KEY_F(8):      addstr_P (PSTR("KEY_F(8)"));    break;
                case KEY_F(9):      addstr_P (PSTR("KEY_F(9)"));    break;
                case KEY_F(10):     addstr_P (PSTR("KEY_F(10)"));   break;
                case KEY_F(11):     addstr_P (PSTR("KEY_F(11)"));   break;
                case KEY_F(12):     addstr_P (PSTR("KEY_F(12)"));   break;
                default:            addch (ch);                     break;
            }

            clrtoeol ();

            if (ch == KEY_ESCAPE)
            {
                PAUSE(500);
                break;
            }
        }

#ifdef unix
        break;
#endif
    }
    endwin ();

    return 0;
}
