/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * @file uart-flash.c
 *
 * Copyright (c) 2011 Robert & Frank Meyer - frank(at)fli4l.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <windows.h>
#include <mmsystem.h>

#include "sndtx.h"

#pragma comment (lib, "C:\\Program Files\\Microsoft SDKs\\Windows\\v6.0A\\Lib\\WinMM.lib")
typedef unsigned char       uint8_t;

#define TRUE                1
#define FALSE               0
#define F_SAMPLES           12000

#define TIMEOUT_SEC         5               // timeout after 5 seconds

#define DATA_BLOCK_START_TICKS  10
#define DATA_BLOCK_END_TICKS     8
#define DATA_BIT_1_TICKS         6
#define DATA_BIT_0_TICKS         4

static volatile int         sound_ready = FALSE;

static void CALLBACK
mycallback (HWAVEOUT hWaveOut, UINT uMsg, DWORD dwInstance, DWORD dwParam1, DWORD dwParam2)
{
    // printf ("mycallback\n");
    if (uMsg == WOM_DONE)
    {
        // printf ("done\n");
        sound_ready = TRUE;
    }
    else if (uMsg == WOM_OPEN)
    {
        // printf ("open\n");
    }
    else if (uMsg == WOM_CLOSE)
    {
        // printf ("close\n");
    }
}


static BYTE *
convert_ticks_to_wave (BYTE * bufp, int ticks)
{
    int idx;

    for (idx = 0; idx < 2; idx++)
    {
        *bufp++ = 255;
    }

    while (idx < ticks)
    {
        *bufp++ = 0;
        idx++;
    }

    return bufp;
}

static BYTE *
convert_bit_to_wave (BYTE * bufp, int bit)
{
    if (bit)                                        // 1: 16 ticks:
    {                                               // 16 values: 31,63,95,127,159,191,223,255,223,191,159,127,95,63,31,0
        bufp = convert_ticks_to_wave (bufp, DATA_BIT_1_TICKS);
    }
    else                                            // 0: 8 ticks:
    {                                               // 8 values: 63,127,191,255,191,127,63,0
        bufp = convert_ticks_to_wave (bufp, DATA_BIT_0_TICKS);
    }
    return bufp;
}

static int
calc_byte_to_wave (int ch)
{
    int idx;
    int bit;
    int sum = 0;

    for (idx = 7; idx >= 0; idx--)
    {
        bit = ch & (1<<idx);

        if (bit)
        {
            sum += DATA_BIT_1_TICKS;
        }
        else
        {
            sum += DATA_BIT_0_TICKS;
        }
    }
    return sum;
}

static BYTE *
convert_block_start_to_wave (BYTE * bufp)
{
    bufp = convert_ticks_to_wave (bufp, DATA_BLOCK_START_TICKS);
    bufp = convert_ticks_to_wave (bufp, DATA_BLOCK_START_TICKS);
    return bufp;
}

static BYTE *
convert_block_end_to_wave (BYTE * bufp)
{
    bufp = convert_ticks_to_wave (bufp, DATA_BLOCK_END_TICKS);
    bufp = convert_ticks_to_wave (bufp, DATA_BLOCK_END_TICKS);
    return bufp;
}

static BYTE *
convert_byte_to_wave (BYTE * bufp, int ch)
{
    int idx;
    int bit;

    for (idx = 7; idx >= 0; idx--)
    {
        bit = ch & (1<<idx);

        bufp = convert_bit_to_wave (bufp, bit);
    }

    return bufp;
}


static long
calc_buffersize (unsigned char * buf, long bufsize)
{
    unsigned char * bufp;
    long            size = 0;

    size += 2 * DATA_BLOCK_START_TICKS;

    size += calc_byte_to_wave ('$');

    for (bufp = buf; bufsize > 0; bufsize--, bufp++)
    {
        size += calc_byte_to_wave (*bufp);
    }

    size += 2 * DATA_BLOCK_END_TICKS;

    return size;
}

static BYTE *
fill_buffer (unsigned char * buf, long bufsize, long wavesize)
{
    BYTE *          wavebuf;
    BYTE *          wavebufp;
    unsigned char * bufp;

    wavebuf = malloc (wavesize);

    if (wavebuf)
    {
        wavebufp = wavebuf;
        wavebufp = convert_block_start_to_wave (wavebufp);

        for (bufp = buf; bufsize > 0; bufsize--, bufp++)
        {
            wavebufp = convert_byte_to_wave (wavebufp, *bufp);
        }

        wavebufp = convert_block_end_to_wave (wavebufp);
    }
    return (wavebuf);
}

int
sound_play (unsigned char * buf, long bufsize)
{
    static HWAVEOUT        outHandle;
    static WAVEFORMATEX    waveFormat;
    static WAVEHDR         hdr;

    unsigned long   result;
    BYTE *          wavebuf;
    long            wavesize;
    int             rtc = ERR;

    ZeroMemory(&waveFormat, sizeof(WAVEFORMATEX));
    ZeroMemory(&hdr, sizeof(WAVEHDR));

    wavesize = calc_buffersize (buf, bufsize);
    wavebuf = fill_buffer (buf, bufsize, wavesize);

    hdr.lpData          = (LPSTR) wavebuf;
    hdr.dwBufferLength  = wavesize;
    hdr.dwBytesRecorded = 0;
    hdr.dwUser          = 0;
    hdr.dwFlags         = 0;
    hdr.dwLoops         = 0;
    hdr.lpNext          = 0;
    hdr.reserved        = 0;

    /* Initialize the WAVEFORMATEX for 8-bit, xx KHz, mono */
    waveFormat.wFormatTag       = WAVE_FORMAT_PCM;
    waveFormat.nChannels        = 1;
    waveFormat.nSamplesPerSec   = F_SAMPLES;
    waveFormat.wBitsPerSample   = 8;
    waveFormat.nBlockAlign      = waveFormat.nChannels * (waveFormat.wBitsPerSample/8);
    waveFormat.nAvgBytesPerSec  = waveFormat.nSamplesPerSec * waveFormat.nBlockAlign;
    waveFormat.cbSize           = 0;

    result = waveOutOpen (&outHandle, WAVE_MAPPER, &waveFormat, (DWORD_PTR) mycallback, 0, CALLBACK_FUNCTION);
    
    if (result == MMSYSERR_NOERROR)
    {
        if (waveOutPrepareHeader (outHandle, &hdr, sizeof (WAVEHDR)) == MMSYSERR_NOERROR)
        {
            if (waveOutWrite (outHandle, &hdr, sizeof (WAVEHDR)) == MMSYSERR_NOERROR)
            {
                int sec;
                int maxsec = wavesize / waveFormat.nSamplesPerSec;

                printf ("buffer size: %ld wave size: %d time: %02d:%02d speed: %d bytes/sec\n",
                        bufsize, wavesize, maxsec / 60, maxsec % 60, bufsize / (maxsec ? maxsec : 1));

                rtc = OK;

                for (sec = 0; ; sec++)
                {
                    if (sound_ready)
                    {
                        break;
                    }
                    printf ("\rTIME: %02d:%02d ETC: %02d:%02d %3d%%",
                            sec / 60, sec % 60, (maxsec - sec) / 60, (maxsec - sec) % 60,
                            maxsec ? ((sec * 100) / maxsec) : 100, maxsec);
                    fflush (stdout);
                    Sleep (1000);
                }
                putchar ('\n');
            }
            else
            {
                fprintf(stderr, "error writing wave data!\r\n");
            }
        }
        else
        {
            fprintf(stderr, "error preparing wave header!\r\n");
        }

        waveOutClose (outHandle);
    }
    else
    {
        fprintf(stderr, "error opening the preferred Digital Audio Out device!\r\n");
    }

	return rtc;
}
