/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * @file sndrx.c
 *
 * Copyright (c) 2011 Robert Meyer, Frank Meyer - frank(at)fli4l.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

#include <stdlib.h>
#include <inttypes.h>
#include <avr/io.h>
#include <avr/interrupt.h>

#include "sndrxconfig.h"
#include "sndrx.h"

#define FALSE                                   0
#define TRUE                                    1

// timing values, see also sndtx.h
#define DATA_BLOCK_START_TICKS                  10                                      // block start
#define DATA_BLOCK_END_TICKS                     8                                      // block end
#define DATA_BIT_1_TICKS                         6                                      // data: 1
#define DATA_BIT_0_TICKS                         4                                      // data: 0
#define MAX_TICKS                               10                                      // max value of ticks, here 10


#define DATA_BLOCK_START_SCAN_TICKS             (3 * DATA_BLOCK_START_TICKS)            // typ. 30
#define DATA_BLOCK_START_SCAN_TICKS_MIN         (DATA_BLOCK_START_SCAN_TICKS - 3)       // min. 27
#define DATA_BLOCK_START_SCAN_TICKS_MAX         (DATA_BLOCK_START_SCAN_TICKS + 2)       // max. 32

#define DATA_BLOCK_END_SCAN_TICKS               (3 * DATA_BLOCK_END_TICKS)              // typ. 24
#define DATA_BLOCK_END_SCAN_TICKS_MIN           (DATA_BLOCK_END_SCAN_TICKS - 3)         // min. 21
#define DATA_BLOCK_END_SCAN_TICKS_MAX           (DATA_BLOCK_END_SCAN_TICKS + 2)         // max. 26

#define DATA_BIT_1_SCAN_TICKS                   (3 * DATA_BIT_1_TICKS)                  // typ. 18
#define DATA_BIT_1_SCAN_TICKS_MIN               (DATA_BIT_1_SCAN_TICKS - 3)             // min. 15
#define DATA_BIT_1_SCAN_TICKS_MAX               (DATA_BIT_1_SCAN_TICKS + 2)             // max. 20

#define DATA_BIT_0_SCAN_TICKS                   (3 * DATA_BIT_0_TICKS)                  // typ. 12
#define DATA_BIT_0_SCAN_TICKS_MIN               (DATA_BIT_0_SCAN_TICKS - 3)             // min.  9
#define DATA_BIT_0_SCAN_TICKS_MAX               (DATA_BIT_0_SCAN_TICKS + 2)             // max. 14

#define MAX_SCAN_TICKS                          (3 * MAX_TICKS)                         // typ. 30
#define MAX_SCAN_TICKS_MAX                      (MAX_SCAN_TICKS + 2)                    // max. 32


static volatile uint8_t                         ringbuf[SNDRX_RINGBUFSIZE];
static volatile uint8_t                         ringbuf_size = 0;
static volatile uint8_t                         sndrx_state = SNDRX_INFO_IDLE;

static volatile uint16_t                        sndrx_timeoutmsec;

/*-----------------------------------------------------------------------------------------------------------------------
 * LOGGING routines, only for debugging!
 *-----------------------------------------------------------------------------------------------------------------------
 */
#if SNDRX_LOGGING == 1
#define LOGBUFSIZE                              250
static volatile uint8_t                         logbuf1[LOGBUFSIZE];
static volatile uint8_t                         logbuf0[LOGBUFSIZE];
static volatile uint8_t                         logbufidx = 0;

#define BAUD                                    9600L
#include <util/setbaud.h>

#ifdef UBRR0H

#define UART0_UBRRH                             UBRR0H
#define UART0_UBRRL                             UBRR0L
#define UART0_UCSRA                             UCSR0A
#define UART0_UCSRB                             UCSR0B
#define UART0_UCSRC                             UCSR0C
#define UART0_UDRE_BIT_VALUE                    (1<<UDRE0)
#define UART0_UCSZ1_BIT_VALUE                   (1<<UCSZ01)
#define UART0_UCSZ0_BIT_VALUE                   (1<<UCSZ00)
#ifdef URSEL0
#define UART0_URSEL_BIT_VALUE                   (1<<URSEL0)
#else
#define UART0_URSEL_BIT_VALUE                   (0)
#endif
#define UART0_TXEN_BIT_VALUE                    (1<<TXEN0)
#define UART0_RXEN_BIT_VALUE                    (1<<RXEN0)
#define UART0_UDR                               UDR0
#define UART0_U2X                               U2X0
#define UART0_RXC                               RXC0

#else

#define UART0_UBRRH                             UBRRH
#define UART0_UBRRL                             UBRRL
#define UART0_UCSRA                             UCSRA
#define UART0_UCSRB                             UCSRB
#define UART0_UCSRC                             UCSRC
#define UART0_UDRE_BIT_VALUE                    (1<<UDRE)
#define UART0_UCSZ1_BIT_VALUE                   (1<<UCSZ1)
#define UART0_UCSZ0_BIT_VALUE                   (1<<UCSZ0)
#ifdef URSEL
#define UART0_URSEL_BIT_VALUE                   (1<<URSEL)
#else
#define UART0_URSEL_BIT_VALUE                   (0)
#endif
#define UART0_TXEN_BIT_VALUE                    (1<<TXEN)
#define UART0_RXEN_BIT_VALUE                    (1<<RXEN)
#define UART0_UDR                               UDR
#define UART0_U2X                               U2X
#define UART0_RXC                               RXC

#endif

static void
sndrx_uart_init (void)
{
    UART0_UBRRH = UBRRH_VALUE;                                                                      // set baud rate
    UART0_UBRRL = UBRRL_VALUE;

#if USE_2X
    UART0_UCSRA |= (1<<UART0_U2X);
#else
    UART0_UCSRA &= ~(1<<UART0_U2X);
#endif

    UART0_UCSRC = UART0_UCSZ1_BIT_VALUE | UART0_UCSZ0_BIT_VALUE | UART0_URSEL_BIT_VALUE;
    UART0_UCSRB |= UART0_TXEN_BIT_VALUE;                                                            // enable UART TX
}

static void
sndrx_uart_putc (unsigned char ch)
{
    while (!(UART0_UCSRA & UART0_UDRE_BIT_VALUE))
    {
        ;
    }

    UART0_UDR = ch;
}

static void
sndrx_uart_puts (char * s)
{
    while (*s)
    {
        sndrx_uart_putc (*s++);
    }
}

void
sndrx_log (void)
{
    uint8_t i;
    char    buf[4];

    for (i = 0; i < LOGBUFSIZE; i++)
    {
        sndrx_uart_puts (itoa (i, buf, 10));
        sndrx_uart_putc (' ');
        sndrx_uart_puts (itoa (logbuf1[i], buf, 10));
        sndrx_uart_putc (' ');
        sndrx_uart_puts (itoa (logbuf0[i], buf, 10));
        sndrx_uart_putc (' ');
        sndrx_uart_puts (itoa (logbuf0[i] + logbuf1[i], buf, 10));
        sndrx_uart_puts ("\r\n");
    }
}

#endif // LOGGING == 1


/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  SNDRX init
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
sndrx_init (void)
{
    OCR1A   =  (F_CPU / (3 * (unsigned) SNDRX_F_SAMPLES)) - 1;                  // compare value: 1/(3*SNDRX_F_SAMPLES) of CPU frequency
    TCCR1B  = (1 << WGM12) | (1 << CS10);                                       // switch CTC Mode on, set prescaler to 1
    TIMSK1  = 1 << OCIE1A;                                                      // OCIE1A: Interrupt by timer compare

    SNDRX_PORT &= ~(1<<SNDRX_BIT);                                              // deactivate pullup
    SNDRX_DDR &= ~(1<<SNDRX_BIT);                                               // set pin to input

#if SNDRX_LOGGING == 1
    sndrx_uart_init ();
#endif
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  SNDRX ISR
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#define MSEC_TOP    (((3 * (unsigned) SNDRX_F_SAMPLES) / 1000) - 1)


ISR(TIMER1_COMPA_vect)
{
#if SNDRX_LOGGING == 1
    static  uint8_t cnt1;
    static  uint8_t cnt0;
#endif
    static  uint8_t busy;
    static  uint8_t last_value;
    static  uint8_t value;
    static  uint8_t do_scan;
    static  uint8_t bit;
    static  uint8_t ch;
    static  uint8_t cnt;
    static  uint8_t ringbuf_stop;
    static  uint8_t startbit_cnt;
    static  uint8_t msec_cnt;

    uint8_t         do_store;

    if (sndrx_timeoutmsec)
    {
        if (msec_cnt < MSEC_TOP)
        {
            msec_cnt++;
        }
        else
        {
            msec_cnt = 0;

            if (sndrx_timeoutmsec)
            {
                sndrx_timeoutmsec--;
            }
        }
    }
    else
    {
        msec_cnt = 0;
    }

    value = input(SNDRX_PIN);

    if (value != last_value)
    {
        if (value)
        {
            if (busy)
            {
#if SNDRX_LOGGING == 1
                if (logbufidx < LOGBUFSIZE)
                {
                    logbuf1[logbufidx] = cnt1;
                    logbuf0[logbufidx] = cnt0;
                    logbufidx++;
                }
#endif

                do_store = FALSE;

                if (cnt >= DATA_BLOCK_START_SCAN_TICKS_MIN && cnt <= DATA_BLOCK_START_SCAN_TICKS_MAX)
                {
                    startbit_cnt++;

                    if (startbit_cnt == 2)
                    {
                        bit = 0;
                        do_scan = TRUE;
                        ch = 0;
                        sndrx_state = SNDRX_INFO_TRANSMIT;
                    }
                }
                else
                {
                    startbit_cnt = 0;

                    if (cnt >= DATA_BIT_0_SCAN_TICKS_MIN && cnt <= DATA_BIT_0_SCAN_TICKS_MAX)
                    {
                        if (do_scan)
                        {
                            do_store = TRUE;
                        }
                    }
                    else if (cnt >= DATA_BIT_1_SCAN_TICKS_MIN && cnt <= DATA_BIT_1_SCAN_TICKS_MAX)
                    {
                        if (do_scan)
                        {
                            ch |= (1 << (7 - bit));
                            do_store = TRUE;
                        }
                    }
                    else if (cnt >= DATA_BLOCK_END_SCAN_TICKS_MIN && cnt <= DATA_BLOCK_END_SCAN_TICKS_MAX)
                    {
                        busy = FALSE;
                        do_scan = FALSE;
                    }
                    else // error
                    {
                        busy = FALSE;

                        if (do_scan)
                        {
                            do_scan = FALSE;
                            sndrx_state = SNDRX_ERROR_FRAME;
                        }
                    }
                }

                if (do_scan && do_store)
                {
                    bit++;

                    if (bit == 8)
                    {
                        if (ringbuf_size < SNDRX_RINGBUFSIZE)
                        {
                            ringbuf[ringbuf_stop++] = ch;

                            if (ringbuf_stop >= SNDRX_RINGBUFSIZE)              // at end of ringbuffer?
                            {                                                   // yes
                                ringbuf_stop = 0;                               // reset to beginning
                            }

                            ringbuf_size++;
                        }
                        else
                        {
                            busy = FALSE;
                            do_scan = FALSE;
                            sndrx_state = SNDRX_ERROR_OVERFLOW;
                        }

                        bit = 0;
                        ch = 0;
                    }
                }
            }

#if SNDRX_LOGGING == 1
            cnt1 = 1;
#endif
            cnt = 1;
            busy = TRUE;
        }
        else
        {
#if SNDRX_LOGGING == 1
            cnt0 = 1;
#endif
            cnt++;
        }

        last_value = value;
    }
    else
    {
#if SNDRX_LOGGING == 1
        if (value)
        {
            cnt1++;
        }
        else
        {
            cnt0++;
        }
#endif

        cnt++;

        if (cnt > MAX_SCAN_TICKS_MAX)
        {
            busy = FALSE;

            if (do_scan)
            {
                do_scan = FALSE;
                sndrx_state = SNDRX_ERROR_FRAME;
            }
            else
            {
                sndrx_state = SNDRX_INFO_IDLE;
            }
        }
    }
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  SNDRX poll
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
int8_t
sndrx_poll (uint8_t * chp, uint16_t timeoutmsec)
{
    static uint8_t  ringbuf_start;
    int8_t          rtc;

    if (ringbuf_size == 0 && sndrx_state >= SNDRX_ERROR_FRAME)
    {                                                   // check only for errors when ringbuffer is empty!
        rtc = -1;
    }
    else
    {
        if (timeoutmsec > 0)
        {
            cli ();
            sndrx_timeoutmsec = timeoutmsec;
            sei ();

            while (sndrx_timeoutmsec && ! ringbuf_size)
            {
                ;
            }
        }

        if (ringbuf_size)
        {
            rtc = 1;
            *chp = ringbuf[ringbuf_start++];            // get character, increment offset

            if (ringbuf_start == SNDRX_RINGBUFSIZE)     // at end of ringbuffer?
            {                                           // yes
                ringbuf_start = 0;                      // reset to beginning
            }

            cli ();
            ringbuf_size--;                             // decrement size
            sei ();
        }
        else
        {
            rtc = 0;
        }
    }
    return rtc;
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  SNDRX status
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
uint8_t
sndrx_status (void)
{
    uint8_t rtc = sndrx_state;

    if (sndrx_state >= SNDRX_ERROR_FRAME)
    {
        sndrx_state = SNDRX_INFO_IDLE;
    }
    return rtc;
}
