/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * @file main.c
 *
 * Copyright (c) 2011 Robert & Frank Meyer - frank(at)fli4l.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "sndtx.h"

#define TRUE                1
#define FALSE               0

#define F_SAMPLES           44100

#define SPM_PAGESIZE        128
#define LINES_PER_PAGE      8

typedef                     unsigned char uint8_t;

static unsigned char        flash[65536];
static unsigned int         address_low     = 0xffff;
static unsigned int         address_high    = 0x0000;

/*------------------------------------------------------------------------------------------------------------------------
 * hex2toi ()
 *------------------------------------------------------------------------------------------------------------------------
 */
static uint8_t
hex2toi (char * p)
{
    uint8_t i = 0;
    uint8_t ch;
    uint8_t idx;

    for (idx = 0; idx < 2; idx++)
    {
        ch = *p++;

        if (ch >= '0' && ch <= '9')
        {
            i |= (ch - '0');
        }
        else if (ch >= 'A' && ch <= 'F')
        {
            i |= (ch - 'A' + 10);
        }
        else if (*p >= 'a' && *p <= 'f')
        {
            i |= (ch - 'a' + 10);
        }

        if (idx < 1)
        {
            i <<= 4;
        }
    }

    return (i);
}

/*------------------------------------------------------------------------------------------------------------------------
 * check_hex_file ()
 *------------------------------------------------------------------------------------------------------------------------
 */
int
check_hex_file (FILE * fp)
{
    char            buf[64];
    int             line_ok;
    unsigned char   len;
    unsigned char   ch;
    unsigned char   idx;
    unsigned char   datalen;
    unsigned int    address;
    unsigned char   addrh;
    unsigned char   addrl;
    unsigned char   rectype;
    char *          dataptr;
    unsigned char   chcksum;
    int             sum = 0;

    while (fgets (buf, 64, fp))
    {
        line_ok = FALSE;

        len = strlen (buf);

        if (len > 2 && *(buf + len - 2) == '\r' && *(buf + len - 1) == '\n')
        {
            *(buf + len - 2) = '\0';
        }

        len = strlen (buf);

        if (buf[0] == ':' && len >= 11)
        {
            datalen = hex2toi (buf +  1);
            addrh   = hex2toi (buf +  3);
            addrl   = hex2toi (buf +  5);
            address = addrh << 8 | addrl;
            rectype = hex2toi (buf + 7);
            dataptr = buf + 9;

            sum = datalen + addrh + addrl + rectype;

            if (len == 9 + 2 * datalen + 2)
            {
                for (idx = 0; idx < datalen; idx++)
                {
                    ch = hex2toi (dataptr);

                    flash[address + idx] = ch;

                    if (address_low > address + idx)
                    {
                        address_low = address + idx;
                    }

                    if (address_high < address + idx)
                    {
                        address_high = address + idx;
                    }

                    sum += ch;
                    dataptr +=2;
                }

                sum = (0x100 - (sum & 0xff)) & 0xff;
                chcksum = hex2toi (dataptr);

                if (sum == chcksum)
                {
                    line_ok = TRUE;
                }
                else
                {
                    fprintf (stderr, "invalid checksum: sum: 0x%02x chcksum: 0x%02x\n", sum, chcksum);
                }
            }
            else
            {
                fprintf (stderr, "invalid len: %d, expected %d\n", len, 12 + 2 * datalen + 2);
            }
        }
        else
        {
            fprintf (stderr, "invalid format, len: %d\n", len);
        }

        if (! line_ok)
        {
            fprintf (stderr, "line '%s' not ok\n", buf);
            return (ERR);
        }
    }

    return (OK);
}

/*------------------------------------------------------------------------------------------------------------------------
 * usage ()
 *------------------------------------------------------------------------------------------------------------------------
 */
static void
usage (char * pgm_name)
{
    fprintf (stderr, "usage: [-f] %s file [outputfile.wav]\n", pgm_name);
    exit (1);
}

/*------------------------------------------------------------------------------------------------------------------------
 * main ()
 *------------------------------------------------------------------------------------------------------------------------
 */
int
main (int argc, char ** argv)
{
    char *          file;
    char *          target = (char *) NULL;
    FILE *          fp;
    unsigned char * buf;
    unsigned char * bufp;
    int             samples;
    long            cnt;
    long            bufsize;
    char *          pgmname;
    char            ch;
    int             do_flash = FALSE;
    int             rtc = 0;

    samples = F_SAMPLES;

    pgmname = argv[0];

    if (argc > 1)
    {
        while (*argv[1] == '-')
        {
            if (! strcmp (argv[1], "-f"))
            {
                do_flash = TRUE;
                argc--;
                argv++;
            }
            else
            {
                usage (pgmname);
            }
        }
    }

    if (argc == 3)
    {
        file = argv[1];
        target = argv[2];
    }
    else if (argc == 2)
    {
        file = argv[1];
    }
    else
    {
        usage (pgmname);
    }

    fp = fopen (file, "rb");

    if (! fp)
    {
        perror (file);
        exit (1);
    }

    if (do_flash)
    {
        char            linebuf[64];
        int             buflen = 0;
        int             page;
        int             page_low;
        int             page_high;
        unsigned char   datalen;
        unsigned int    address;
        unsigned char   rectype;
        int             len;
        int             line;
        int             rtc = 0;
        int             sum = 0;
        int             idx;

        if (check_hex_file (fp) != OK)
        {
            fclose (fp);
            exit (1);
        }

        page_low    = address_low / SPM_PAGESIZE;
        page_high   = address_high / SPM_PAGESIZE + 1;

        bufsize = 3 * (page_high - page_low) * SPM_PAGESIZE + 64;

        buf = malloc (bufsize);

        printf ("low = 0x%04x high = 0x%04x\n", address_low, address_high);

        if (! buf)
        {
            perror ("malloc");
            exit (1);
        }

        buf[buflen++] = '$';

        for (page = page_low; page < page_high; page++)
        {
            for (line = 0; line < LINES_PER_PAGE; line++)
            {
                sum = 0;

                datalen = SPM_PAGESIZE / LINES_PER_PAGE;
                address = page * SPM_PAGESIZE + line * datalen;
                rectype = 0x00;
                sprintf (linebuf, ":%02X%04X%02X", datalen, address, rectype);

                sum = datalen + (address >> 8) + (address & 0xFF) + rectype;

                len = strlen (linebuf);

                for (idx = 0; idx < datalen; idx++)
                {
                    sprintf (linebuf + len + 2 * idx, "%02X", flash[address + idx]);
                    sum += flash[address + idx];
                }

                sum = (0x100 - (sum & 0xff)) & 0xff;
                sprintf (linebuf + len + 2 * datalen, "%02X", sum);

                sprintf (buf + buflen, "%s\r\n", linebuf);
                buflen += strlen (linebuf) + 2;
            }
        }

        strcpy (linebuf, ":00000001FF");
        sprintf (buf + buflen, "%s\r\n", linebuf);        // EOF
        buflen += strlen (linebuf) + 2;
        buf[buflen] = '\0';

        bufsize = buflen;
    }
    else
    {
        fseek (fp, 0L, SEEK_END);
        bufsize = ftell (fp);
        rewind (fp);
        buf = malloc (bufsize);

        if (! buf)
        {
            perror ("malloc");
            exit (1);
        }

        for (bufp = buf, cnt = 0; cnt < bufsize && (ch = getc (fp)) != EOF; bufp++, cnt++)
        {
            *bufp = ch;
        }

        fclose (fp);
    }

    if (sound_play (buf, bufsize, samples, target) == ERR)
    {
        rtc = 1; 
    }

    free (buf);
    return (rtc);
}
