/*---------------------------------------------------------------------------------------------------------
 * sndrx-bootloader.c
 *
 * Copyright (c) 2011 Robert Meyer, Frank Meyer
 *
 * ATMEGA168 @ 8 MHz
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------
 */

#include <stdlib.h>
#include <stdio.h>
#include <avr/io.h>
#include <avr/boot.h>
#include <avr/interrupt.h>
#include <avr/eeprom.h>
#include <avr/wdt.h>
#include <util/delay.h>
#include <inttypes.h>

#include "sndrxconfig.h"
#include "sndrx.h"

#define TRUE    1
#define FALSE   0

#define USE_LED         1

#if USE_LED == 1

/*-----------------------------------------------------------------------------------------------------------------------
 * Status LED, change here:
 *-----------------------------------------------------------------------------------------------------------------------
 */
#define LED_PORT        PORTD
#define LED_DDR         DDRD
#define LED             6

#define LED_ON          LED_PORT |= 1<<LED
#define LED_OFF         LED_PORT &= ~(1<<LED)

/*-----------------------------------------------------------------------------------------------------------------------
 * Initialize LED port
 *-----------------------------------------------------------------------------------------------------------------------
 */
static void
led_init (void)
{
    LED_OFF;
    LED_DDR |= (1<<LED);
}

#endif // USE_LED == 1

/*-----------------------------------------------------------------------------------------------------------------------
 * Flash: erase page
 *-----------------------------------------------------------------------------------------------------------------------
 */
#define boot_program_page_erase(pageaddr)           \
{                                                   \
    eeprom_busy_wait ();                            \
    cli ();                                         \
    boot_page_erase ((uint32_t) (pageaddr));        \
    sei ();                                         \
    boot_spm_busy_wait ();                          \
}

/*-----------------------------------------------------------------------------------------------------------------------
 * Flash: fill page word by word
 *-----------------------------------------------------------------------------------------------------------------------
 */
#define boot_program_page_fill(byteaddr, word)      \
{                                                   \
    cli ();                                         \
    boot_page_fill ((uint32_t) (byteaddr), word);   \
    sei ();                                         \
}

/*-----------------------------------------------------------------------------------------------------------------------
 * Flash: write page
 *-----------------------------------------------------------------------------------------------------------------------
 */
#define boot_program_page_write(pageaddr)           \
{                                                   \
    cli ();                                         \
    boot_page_write ((uint32_t) (pageaddr));        \
    sei ();                                         \
    boot_spm_busy_wait ();                          \
    boot_rww_enable ();                             \
}


static uint8_t
boot_update_flash (void)
{
    uint8_t         start_addr_h;
    uint8_t         start_addr_l;
    uint8_t         page_buf[2];
    uint16_t        start_addr;
    uint8_t         idx;
    int8_t          rtc;
    uint16_t        w;

#if USE_LED == 1
    LED_OFF;
#endif

    if (sndrx_poll (&start_addr_h, 20) <= 0 ||
        sndrx_poll (&start_addr_l, 20) <= 0)
    {
        return FALSE;
    }

    for (start_addr = (((uint16_t) start_addr_h) << 8) | start_addr_l; ; start_addr += SPM_PAGESIZE)
    {
#if USE_LED == 1
        LED_OFF;
#endif
        for (idx = 0; idx < SPM_PAGESIZE; idx++)
        {
            rtc = sndrx_poll (page_buf + (idx & 0x01), 20);
            if (rtc < 0)
            {
                return FALSE;
            }
            else if (rtc == 0)
            {
                break;
            }

            if (idx == 0)
            {
                boot_program_page_erase (start_addr);                           // erase at beginning of page, but after 1 char read
            }

            if (idx & 0x01)                                                     // odd?
            {
                w = page_buf[0] | (page_buf[1] << 8);
                boot_program_page_fill ((uint32_t) (start_addr + idx - 1), w);
            }

        }

#if USE_LED == 1
        LED_ON;
#endif
        if (idx == 0)                                                           // we are ready
        {
            break;
        }
        else if (idx < SPM_PAGESIZE)                                            // error: page not completed
        {
            return FALSE;
        }

        boot_program_page_write ((uint32_t) start_addr);
    }

    return TRUE;
}

/*-----------------------------------------------------------------------------------------------------------------------
 * main function
 *-----------------------------------------------------------------------------------------------------------------------
 */
int
main (void)
{
    uint8_t     ch;
    uint8_t     temp;

    sndrx_init ();                                                          // initialize SOUNDRX

#if USE_LED == 1
    led_init();                                                             // initialize LED port
    LED_ON;                                                                 // switch LED on
#endif

    temp = MCUCR;                                                           // set interrupt vectors to bootloader section
    MCUCR = temp | (1<<IVCE);
    MCUCR = temp | (1<<IVSEL);

    sei ();                                                                 // enable interrupts

    if (sndrx_poll (&ch, 3000) == 1 && ch == '$')                           // wait 3 seconds for the dollar character...
    {
#if USE_LED == 1
        LED_OFF;                                                            // got it, switch LED off
#endif
        boot_update_flash ();                                               // flash now
    }

#if USE_LED == 1
    LED_OFF;                                                               // switch LED off
#endif

    cli ();                                                                 // disable interrupts

    temp = MCUCR;                                                           // reset interrupt vectors (set to application section)
    MCUCR = temp | (1<<IVCE);
    MCUCR = temp & ~(1<<IVSEL);

    asm volatile("jmp 0x0000");
    return 0;
}
