/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * @file sndrx.c
 *
 * Copyright (c) 2011 Robert Meyer, Frank Meyer - frank(at)fli4l.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

#include <stdlib.h>
#include <inttypes.h>
#include <avr/io.h>
#include <avr/interrupt.h>

#include "sndrxconfig.h"
#include "sndrx.h"

#ifdef SNDRX_AS_INCLUDE                                                                 // if used as include....
#define SNDRX_STATIC   static                                                           // we can declare functions as static
#else                                                                                   // this saves program space
#define SNDRX_STATIC
#endif

#define FALSE                                   0
#define TRUE                                    1

// timing values, see also sndtx.h
#define DATA_BLOCK_START_TICKS                 13                                       // block start
#define DATA_BLOCK_END_TICKS                   10                                       // block end
#define DATA_BIT_1_TICKS                        7                                       // data: 1
#define DATA_BIT_0_TICKS                        4                                       // data: 0


#define DATA_BLOCK_START_SCAN_TICKS             (DATA_BLOCK_START_TICKS)                // typ. 13
#define DATA_BLOCK_START_SCAN_TICKS_MIN         (DATA_BLOCK_START_SCAN_TICKS - 1)       // min. 12
#define DATA_BLOCK_START_SCAN_TICKS_MAX         (DATA_BLOCK_START_SCAN_TICKS + 2)       // max. 15 !!!

#define DATA_BLOCK_END_SCAN_TICKS               (DATA_BLOCK_END_TICKS)                  // typ. 10
#define DATA_BLOCK_END_SCAN_TICKS_MIN           (DATA_BLOCK_END_SCAN_TICKS - 1)         // min.  9
#define DATA_BLOCK_END_SCAN_TICKS_MAX           (DATA_BLOCK_END_SCAN_TICKS + 1)         // max. 11

#define DATA_BIT_1_SCAN_TICKS                   (DATA_BIT_1_TICKS)                      // typ.  7
#define DATA_BIT_1_SCAN_TICKS_MIN               (DATA_BIT_1_SCAN_TICKS - 1)             // min.  6
#define DATA_BIT_1_SCAN_TICKS_MAX               (DATA_BIT_1_SCAN_TICKS + 1)             // max.  8

#define DATA_BIT_0_SCAN_TICKS                   (DATA_BIT_0_TICKS)                      // typ.  4
#define DATA_BIT_0_SCAN_TICKS_MIN               (DATA_BIT_0_SCAN_TICKS - 1)             // min.  3
#define DATA_BIT_0_SCAN_TICKS_MAX               (DATA_BIT_0_SCAN_TICKS + 1)             // max.  5

#define MAX_SCAN_TICKS                          DATA_BLOCK_START_SCAN_TICKS             // typ. 13
#define MAX_SCAN_TICKS_MAX                      DATA_BLOCK_START_SCAN_TICKS_MAX         // max. 15


#if SNDRX_RINGBUFSIZE == 0
static volatile uint8_t                         inputch;
static volatile uint8_t                         inputch_stored;
#else
static volatile uint8_t                         ringbuf[SNDRX_RINGBUFSIZE];

static volatile uint8_t                         ringbuf_size;
static volatile uint8_t                         ringbuf_stop;
static volatile uint8_t                         ringbuf_start;
#endif

static volatile uint8_t                         sndrx_state = SNDRX_INFO_IDLE;

static volatile uint16_t                        sndrx_timeoutmsec;

/*-----------------------------------------------------------------------------------------------------------------------
 * LOGGING routines, ONLY for debugging!
 *-----------------------------------------------------------------------------------------------------------------------
 */
#if SNDRX_LOGGING == 1
#define LOGBUFSIZE                              250
static volatile uint8_t                         logbuf1[LOGBUFSIZE];
static volatile uint8_t                         logbuf0[LOGBUFSIZE];
static volatile uint8_t                         logbufidx = 0;

#define BAUD                                    19200L
#include <util/setbaud.h>

#ifdef UBRR0H

#define UART0_UBRRH                             UBRR0H
#define UART0_UBRRL                             UBRR0L
#define UART0_UCSRA                             UCSR0A
#define UART0_UCSRB                             UCSR0B
#define UART0_UCSRC                             UCSR0C
#define UART0_UDRE_BIT_VALUE                    (1<<UDRE0)
#define UART0_UCSZ1_BIT_VALUE                   (1<<UCSZ01)
#define UART0_UCSZ0_BIT_VALUE                   (1<<UCSZ00)
#ifdef URSEL0
#define UART0_URSEL_BIT_VALUE                   (1<<URSEL0)
#else
#define UART0_URSEL_BIT_VALUE                   (0)
#endif
#define UART0_TXEN_BIT_VALUE                    (1<<TXEN0)
#define UART0_RXEN_BIT_VALUE                    (1<<RXEN0)
#define UART0_UDR                               UDR0
#define UART0_U2X                               U2X0
#define UART0_RXC                               RXC0

#else

#define UART0_UBRRH                             UBRRH
#define UART0_UBRRL                             UBRRL
#define UART0_UCSRA                             UCSRA
#define UART0_UCSRB                             UCSRB
#define UART0_UCSRC                             UCSRC
#define UART0_UDRE_BIT_VALUE                    (1<<UDRE)
#define UART0_UCSZ1_BIT_VALUE                   (1<<UCSZ1)
#define UART0_UCSZ0_BIT_VALUE                   (1<<UCSZ0)
#ifdef URSEL
#define UART0_URSEL_BIT_VALUE                   (1<<URSEL)
#else
#define UART0_URSEL_BIT_VALUE                   (0)
#endif
#define UART0_TXEN_BIT_VALUE                    (1<<TXEN)
#define UART0_RXEN_BIT_VALUE                    (1<<RXEN)
#define UART0_UDR                               UDR
#define UART0_U2X                               U2X
#define UART0_RXC                               RXC

#endif

static void
sndrx_uart_init (void)
{
    UART0_UBRRH = UBRRH_VALUE;                                                                      // set baud rate
    UART0_UBRRL = UBRRL_VALUE;

#if USE_2X
    UART0_UCSRA |= (1<<UART0_U2X);
#else
    UART0_UCSRA &= ~(1<<UART0_U2X);
#endif

    UART0_UCSRC = UART0_UCSZ1_BIT_VALUE | UART0_UCSZ0_BIT_VALUE | UART0_URSEL_BIT_VALUE;
    UART0_UCSRB |= UART0_TXEN_BIT_VALUE;                                                            // enable UART TX
}

static void
sndrx_uart_putc (unsigned char ch)
{
    while (!(UART0_UCSRA & UART0_UDRE_BIT_VALUE))
    {
        ;
    }

    UART0_UDR = ch;
}

static void
sndrx_uart_puts (char * s)
{
    while (*s)
    {
        sndrx_uart_putc (*s++);
    }
}

SNDRX_STATIC void
sndrx_log (void)
{
    uint8_t i;
    char    buf[4];

    for (i = 0; i < LOGBUFSIZE; i++)
    {
        sndrx_uart_puts (itoa (i, buf, 10));
        sndrx_uart_putc (' ');
        sndrx_uart_puts (itoa (logbuf0[i], buf, 10));
        sndrx_uart_putc (' ');
        sndrx_uart_puts (itoa (logbuf1[i], buf, 10));
        sndrx_uart_putc (' ');
        sndrx_uart_puts (itoa (logbuf0[i] + logbuf1[i], buf, 10));
        sndrx_uart_puts ("\r\n");
    }
}

#endif // LOGGING == 1


/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  SNDRX init
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
SNDRX_STATIC void
sndrx_init (void)
{
    OCR1A   =  (F_CPU / ((unsigned) SNDRX_F_SAMPLES)) - 1;                      // compare value: 1/(3*SNDRX_F_SAMPLES) of CPU frequency
    TCCR1B  = (1 << WGM12) | (1 << CS10);                                       // switch CTC Mode on, set prescaler to 1
    TIMSK1  = 1 << OCIE1A;                                                      // OCIE1A: Interrupt by timer compare

    SNDRX_PORT |= 1<<SNDRX_BIT;                                                 // activate pullup
    SNDRX_DDR &= ~(1<<SNDRX_BIT);                                               // set pin to input

#if SNDRX_LOGGING == 1
    sndrx_uart_init ();
#endif
}


/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  SNDRX ISR
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#define MSEC_TOP    ((((unsigned) SNDRX_F_SAMPLES) / 1000) - 1)

ISR(TIMER1_COMPA_vect)
{
#if SNDRX_LOGGING == 1
    static  uint8_t cnt1;
    static  uint8_t cnt0;
#endif
    static  uint8_t last_value;
    static  uint8_t value;
    static  uint8_t bit;
    static  uint8_t ch;
    static  uint8_t cnt;
    static  uint8_t startbit_cnt;
    static  uint8_t msec_cnt;

    uint8_t         state = sndrx_state;
    uint16_t        msec = sndrx_timeoutmsec;

    if (msec)
    {
        if (msec_cnt < MSEC_TOP)
        {
            msec_cnt++;
        }
        else
        {
            msec_cnt = 0;

            if (msec)
            {
                sndrx_timeoutmsec = msec - 1;
            }
        }
    }
    else
    {
        msec_cnt = 0;
    }

    value = input(SNDRX_PIN);

    if (value != last_value)
    {
        if (value)
        {
#if SNDRX_LOGGING == 1
            if (logbufidx < LOGBUFSIZE)
            {
                logbuf1[logbufidx] = cnt1;
                logbuf0[logbufidx] = cnt0;
                logbufidx++;
            }
#endif
            if (cnt < DATA_BIT_0_SCAN_TICKS_MIN || cnt > DATA_BLOCK_START_SCAN_TICKS_MAX)
            {
                if (state == SNDRX_INFO_TRANSMIT)
                {
                    startbit_cnt = 0;

                    state = SNDRX_ERROR_FRAME;
#if SNDRX_RINGBUFSIZE == 0
                    inputch_stored = FALSE;
#else
                    ringbuf_start = 0;
                    ringbuf_stop = 0;
                    ringbuf_size = 0;
#endif
                }
            }
            else if (cnt >= DATA_BLOCK_START_SCAN_TICKS_MIN)                            // start bit
            {
                startbit_cnt++;

                if (startbit_cnt == 2)
                {
                    bit = 0;
                    state = SNDRX_INFO_TRANSMIT;
                    ch = 0;
                }
            }
            else
            {
                startbit_cnt = 0;

                if (state == SNDRX_INFO_TRANSMIT)
                {
                    if (cnt <= DATA_BIT_0_SCAN_TICKS_MAX)                                   // 0-bit
                    {
                        ch <<= 1;
                    }
                    else if (cnt <= DATA_BIT_1_SCAN_TICKS_MAX)                              // 1-bit
                    {
                        ch <<= 1;
                        ch |= 1;
                    }
                    else // if (cnt <= DATA_BLOCK_END_SCAN_TICKS_MAX)                       // stop bit
                    {
                        state = SNDRX_INFO_IDLE;
                    }
                }

                if (state == SNDRX_INFO_TRANSMIT)
                {
                    bit++;

                    if (bit == 8)
                    {
#if SNDRX_RINGBUFSIZE == 0
                        if (inputch_stored)
                        {
                            state = SNDRX_ERROR_OVERFLOW;
                        }
                        else
                        {
                            inputch = ch;
                            inputch_stored = TRUE;
                        }
#else
                        if (ringbuf_size < SNDRX_RINGBUFSIZE)
                        {
                            uint8_t stop = ringbuf_stop;

                            ringbuf[stop++] = ch;

                            if (stop >= SNDRX_RINGBUFSIZE)                      // at end of ringbuffer?
                            {                                                   // yes
                                stop = 0;                                       // reset to beginning
                            }

                            ringbuf_stop = stop;
                            ringbuf_size++;
                        }
                        else
                        {
                            state = SNDRX_ERROR_OVERFLOW;
                            ringbuf_start = 0;
                            ringbuf_stop = 0;
                            ringbuf_size = 0;
                        }
#endif

                        bit = 0;
                        ch = 0;
                    }
                }
            }


#if SNDRX_LOGGING == 1
            cnt1 = 1;
#endif
            cnt = 1;
        }
        else
        {
#if SNDRX_LOGGING == 1
            cnt0 = 1;
#endif
            cnt++;
        }

        last_value = value;
    }
    else
    {
#if SNDRX_LOGGING == 1
        if (value)
        {
            cnt1++;
        }
        else
        {
            cnt0++;
        }
#endif

        cnt++;

        if (state == SNDRX_INFO_TRANSMIT && cnt > MAX_SCAN_TICKS_MAX)
        {
            state = SNDRX_ERROR_TIMEOUT;
#if SNDRX_RINGBUFSIZE == 0
            inputch_stored = FALSE;
#else
            ringbuf_start = 0;
            ringbuf_stop = 0;
            ringbuf_size = 0;
#endif
        }
    }

    sndrx_state = state;
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  SNDRX poll: poll with timeout (msec)
 *
 *  Return value    Remark                                          sndrx_status() will then return:
 *
 *      0           no character available within timeout           SNDRX_INFO_IDLE
 *      1           one character available, stored in (*chp)       SNDRX_INFO_TRANSMIT
 *     -1           error occured                                   SNDRX_ERROR_xxx
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
SNDRX_STATIC int8_t
sndrx_poll (uint8_t * chp, uint16_t timeoutmsec)
{
    int8_t          rtc;

    if (sndrx_state >= SNDRX_ERROR_FRAME)
    {
        rtc = -1;
    }
    else
    {
        if (timeoutmsec > 0)
        {
            cli ();
            sndrx_timeoutmsec = timeoutmsec;
            sei ();

#if SNDRX_RINGBUFSIZE == 0
            while (sndrx_timeoutmsec && ! inputch_stored)
#else
            while (sndrx_timeoutmsec && ! ringbuf_size)
#endif
            {
                ;
            }
        }

#if SNDRX_RINGBUFSIZE == 0
        if (inputch_stored)
        {
            rtc = 1;

            *chp = inputch;                             // get character, increment offset
            cli ();
            inputch_stored = FALSE;
            sei ();
        }
#else
        if (ringbuf_size)
        {
            uint8_t start;

            rtc = 1;

            cli ();
            start = ringbuf_start;
            *chp = ringbuf[start++];                    // get character, increment offset

            if (start == SNDRX_RINGBUFSIZE)             // at end of ringbuffer?
            {                                           // yes
                start = 0;                              // reset to beginning
            }

            ringbuf_start = start;
            ringbuf_size--;                             // decrement size
            sei ();
        }
#endif
        else
        {
            rtc = 0;
        }
    }
    return rtc;
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  SNDRX status
 *
 *  Return values:
 *
 *    SNDRX_INFO_IDLE                         INFO:  sndrx is idle
 *    SNDRX_INFO_TRANSMIT                     INFO:  sender is transmitting
 *    SNDRX_ERROR_FRAME                       ERROR: frame error occured
 *    SNDRX_ERROR_OVERFLOW                    ERROR: ringbuffer overflow
 *    SNDRX_ERROR_TIMEOUT                     ERROR: timeout occured (buffer underrun)
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  SNDRX stop transmission
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
SNDRX_STATIC void
sndrx_stop (void)
{
    cli ();
    sndrx_state = SNDRX_INFO_IDLE;
    sei ();
}

#ifdef SNDRX_AS_INCLUDE

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  SNDRX done: deactivate timer used by sndrx - only relevant if bootloader
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
SNDRX_STATIC void
sndrx_done (void)
{
    TIMSK1  = 0;                                                                // reset OCIE1A
    TCCR1B  = 0;                                                                // reset TCCR1B
    OCR1A   = 0;                                                                // reset OCR1A
    SNDRX_PORT &= ~(1<<SNDRX_BIT);                                              // deactivate pullup
}

#else

uint8_t
sndrx_status (void)
{
    uint8_t rtc = sndrx_state;

    if (sndrx_state >= SNDRX_ERROR_FRAME)
    {
        cli ();
        sndrx_state = SNDRX_INFO_IDLE;
        sei ();
    }
    return rtc;
}

#endif
