/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * @file sndrx.c
 *
 * Copyright (c) 2011 Robert Meyer, Frank Meyer - frank(at)fli4l.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

#include <stdlib.h>
#include <inttypes.h>
#include <avr/io.h>
#include <avr/interrupt.h>

#include "sndrxconfig.h"
#include "sndrx.h"

#ifdef SNDRX_AS_INCLUDE                                                                 // if used as include....
#define SNDRX_STATIC   static                                                           // we can declare functions as static
#else                                                                                   // this saves program space
#define SNDRX_STATIC
#endif

#define FALSE                                   0
#define TRUE                                    1

#define F_INTERRUPTS                            1000                                    // 1000 interrupts/sec by timer1
#define PRESCALER                               64                                      // prescaler for timer1

#define MIN_TOLERANCE_00                        1.0                           // -0%
#define MAX_TOLERANCE_00                        1.0                           // +0%

#define MIN_TOLERANCE_05                        0.95                          // -5%
#define MAX_TOLERANCE_05                        1.05                          // +5%

#define MIN_TOLERANCE_10                        0.9                           // -10%
#define MAX_TOLERANCE_10                        1.1                           // +10%

#define MIN_TOLERANCE_15                        0.85                          // -15%
#define MAX_TOLERANCE_15                        1.15                          // +15%

#define MIN_TOLERANCE_20                        0.8                           // -20%
#define MAX_TOLERANCE_20                        1.2                           // +20%

#define MIN_TOLERANCE_30                        0.7                           // -30%
#define MAX_TOLERANCE_30                        1.3                           // +30%

#define MIN_TOLERANCE_40                        0.6                           // -40%
#define MAX_TOLERANCE_40                        1.4                           // +40%

#define MIN_TOLERANCE_50                        0.5                           // -50%
#define MAX_TOLERANCE_50                        1.5                           // +50%

#define MIN_TOLERANCE_60                        0.4                           // -60%
#define MAX_TOLERANCE_60                        1.6                           // +60%

#define MIN_TOLERANCE_70                        0.3                           // -70%
#define MAX_TOLERANCE_70                        1.7                           // +70%

// timing values, see also sndtx.h
#define DATA_BIT_0_TICKS                        2                                       // data: 0
#define DATA_BIT_1_TICKS                        3                                       // data: 1
#define DATA_BLOCK_END_TICKS                    6                                       // block end
#define DATA_BLOCK_START_TICKS                  9                                       // block start

// F_CPU     8000000
// Prescaler	  64
// Divisor         3       4       5       6       7       8       9      10
// N_SAMPLES   41667   31250   25000   20833   17857   15625   13889   12500
//     2           6       8      10      12      14      16      18      20
//     3           9      12      15      18      21      24      27      30
//     6          18      24      30      36      42      48      54      60
//     9          27      36      45      54      63      72      81      90
// Min speed:   1736    1302    1042     868     744     651     579     521
// Max speed:   2604    1953    1563    1302    1116     977     868     781
// Avg speed:   2083    1563    1250    1042     893     781     694     625


#define DATA_BIT_0_SCAN_TICKS                   (uint8_t) ((F_CPU / (float) PRESCALER / SNDRX_F_SAMPLES) * DATA_BIT_0_TICKS  + 0.5)
#define DATA_BIT_1_SCAN_TICKS                   (uint8_t) ((F_CPU / (float) PRESCALER / SNDRX_F_SAMPLES) * DATA_BIT_1_TICKS + 0.5)
#define DATA_BLOCK_END_SCAN_TICKS               (uint8_t) ((F_CPU / (float) PRESCALER / SNDRX_F_SAMPLES) * DATA_BLOCK_END_TICKS + 0.5)
#define DATA_BLOCK_START_SCAN_TICKS             (uint8_t) ((F_CPU / (float) PRESCALER / SNDRX_F_SAMPLES) * DATA_BLOCK_START_TICKS + 0.5)

#define DATA_BIT_0_SCAN_TICKS_MIN               (uint8_t) (((MIN_TOLERANCE_20 * F_CPU) / (float) PRESCALER / SNDRX_F_SAMPLES) * DATA_BIT_0_TICKS  + 0.5)
#define DATA_BIT_0_SCAN_TICKS_MAX               (uint8_t) (((MAX_TOLERANCE_20 * F_CPU) / (float) PRESCALER / SNDRX_F_SAMPLES) * DATA_BIT_0_TICKS  + 0.5)
#define DATA_BIT_1_SCAN_TICKS_MIN               (uint8_t) (((MIN_TOLERANCE_20 * F_CPU) / (float) PRESCALER / SNDRX_F_SAMPLES) * DATA_BIT_1_TICKS  + 0.5)
#define DATA_BIT_1_SCAN_TICKS_MAX               (uint8_t) (((MAX_TOLERANCE_20 * F_CPU) / (float) PRESCALER / SNDRX_F_SAMPLES) * DATA_BIT_1_TICKS  + 0.5)
#define DATA_BLOCK_END_SCAN_TICKS_MIN           (uint8_t) (((MIN_TOLERANCE_20 * F_CPU) / (float) PRESCALER / SNDRX_F_SAMPLES) * DATA_BLOCK_END_TICKS  + 0.5)
#define DATA_BLOCK_END_SCAN_TICKS_MAX           (uint8_t) (((MAX_TOLERANCE_20 * F_CPU) / (float) PRESCALER / SNDRX_F_SAMPLES) * DATA_BLOCK_END_TICKS  + 0.5)
#define DATA_BLOCK_START_SCAN_TICKS_MIN         (uint8_t) (((MIN_TOLERANCE_20 * F_CPU) / (float) PRESCALER / SNDRX_F_SAMPLES) * DATA_BLOCK_START_TICKS  + 0.5)
#define DATA_BLOCK_START_SCAN_TICKS_MAX         (uint8_t) (((MAX_TOLERANCE_20 * F_CPU) / (float) PRESCALER / SNDRX_F_SAMPLES) * DATA_BLOCK_START_TICKS  + 0.5)

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * Port pins
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#if SNDRX_PRT == 'A'
#define SNDRX_PORT                          PORTA
#define SNDRX_DDR                           DDRA
#define SNDRX_PIN                           PINA
#elif SNDRX_PRT == 'B'
#define SNDRX_PORT                          PORTB
#define SNDRX_DDR                           DDRB
#define SNDRX_PIN                           PINB
#elif SNDRX_PRT == 'C'
#define SNDRX_PORT                          PORTC
#define SNDRX_DDR                           DDRC
#define SNDRX_PIN                           PINC
#elif SNDRX_PRT == 'D'
#define SNDRX_PORT                          PORTD
#define SNDRX_DDR                           DDRD
#define SNDRX_PIN                           PIND
#elif SNDRX_PRT == 'E'
#define SNDRX_PORT                          PORTE
#define SNDRX_DDR                           DDRE
#define SNDRX_PIN                           PINE
#elif SNDRX_PRT == 'F'
#define SNDRX_PORT                          PORTF
#define SNDRX_DDR                           DDRF
#define SNDRX_PIN                           PINF
#elif SNDRX_PRT == 'G'
#define SNDRX_PORT                          PORTG
#define SNDRX_DDR                           DDRG
#define SNDRX_PIN                           PING
#elif SNDRX_PRT == 'H'
#define SNDRX_PORT                          PORTH
#define SNDRX_DDR                           DDRH
#define SNDRX_PIN                           PINH
#elif SNDRX_PRT == 'I'
#define SNDRX_PORT                          PORTI
#define SNDRX_DDR                           DDRI
#define SNDRX_PIN                           PINI
#elif SNDRX_PRT == 'J'
#define SNDRX_PORT                          PORTJ
#define SNDRX_DDR                           DDRJ
#define SNDRX_PIN                           PINJ
#elif SNDRX_PRT == 'K'
#define SNDRX_PORT                          PORTK
#define SNDRX_DDR                           DDRK
#define SNDRX_PIN                           PINK
#endif

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * PCINT for ATmega48/88/168/328
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#if defined (__AVR_ATmega48__)   \
 || defined (__AVR_ATmega88__)   \
 || defined (__AVR_ATmega168__)  \
 || defined (__AVR_ATmega168__)  \
 || defined (__AVR_ATmega168P__) \
 || defined (__AVR_ATmega328__)  \
 || defined (__AVR_ATmega328P__)

#if SNDRX_PRT == 'B'
#define SNDRX_MSK                           PCMSK0 
#define SNDRX_PCIE                          PCIE0
#define SNDRX_PCINT_vect                    PCINT0_vect
#elif SNDRX_PRT == 'C'
#define SNDRX_MSK                           PCMSK1
#define SNDRX_PCIE                          PCIE1
#define SNDRX_PCINT_vect                    PCINT1_vect
#elif SNDRX_PRT == 'D'
#define SNDRX_MSK                           PCMSK2
#define SNDRX_PCIE                          PCIE2
#define SNDRX_PCINT_vect                    PCINT2_vect
#else
#error SNDRX_PRT undeclared or wrong value!
#endif

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * PCINT for ATmega162
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#elif defined (__AVR_ATmega162__)

#if SNDRX_PRT == 'A'
#define SNDRX_MSK                           PCMSK0 
#define SNDRX_PCIE                          PCIE0
#define SNDRX_PCINT_vect                    PCINT0_vect
#elif SNDRX_PRT == 'C'
#define SNDRX_MSK                           PCMSK1
#define SNDRX_PCIE                          PCIE1
#define SNDRX_PCINT_vect                    PCINT1_vect
#else
#error SNDRX_PRT undeclared or wrong value!
#endif

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * PCINT for ATmega164/324/644/1284
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#elif defined (__AVR_ATmega164P__)  \
   || defined (__AVR_ATmega324P__)  \
   || defined (__AVR_ATmega644__)   \
   || defined (__AVR_ATmega644P__)  \
   || defined (__AVR_ATmega1284P__)
 
#if SNDRX_PRT == 'A'
#define SNDRX_MSK                           PCMSK0 
#define SNDRX_PCIE                          PCIE0
#define SNDRX_PCINT_vect                    PCINT0_vect
#elif SNDRX_PRT == 'B'
#define SNDRX_MSK                           PCMSK1
#define SNDRX_PCIE                          PCIE1
#define SNDRX_PCINT_vect                    PCINT1_vect
#elif SNDRX_PRT == 'C'
#define SNDRX_MSK                           PCMSK2
#define SNDRX_PCIE                          PCIE2
#define SNDRX_PCINT_vect                    PCINT2_vect
#elif SNDRX_PRT == 'D'
#define SNDRX_MSK                           PCMSK3
#define SNDRX_PCIE                          PCIE3
#define SNDRX_PCINT_vect                    PCINT3_vect
#else
#error SNDRX_PRT undeclared or wrong value!
#endif

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * PCINT for ATmega640/1280/1281/2560/2561
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#elif defined (__AVR_ATmega640__)   \
   || defined (__AVR_ATmega1280__)  \
   || defined (__AVR_ATmega1281__)  \
   || defined (__AVR_ATmega2560__)  \
   || defined (__AVR_ATmega2561__)
 
#if SNDRX_PRT == 'B'
#define SNDRX_MSK                           PCMSK0 
#define SNDRX_PCIE                          PCIE0
#define SNDRX_PCINT_vect                    PCINT0_vect
#elif SNDRX_PRT == 'J'
#define SNDRX_MSK                           PCMSK1
#define SNDRX_PCIE                          PCIE1
#define SNDRX_PCINT_vect                    PCINT1_vect
#elif SNDRX_PRT == 'K'
#define SNDRX_MSK                           PCMSK2
#define SNDRX_PCIE                          PCIE2
#define SNDRX_PCINT_vect                    PCINT2_vect
#else
#error SNDRX_PRT undeclared or wrong value!
#endif

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * PCINT for unknown microcontroller
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#else
#error this microcontroller is not supported by SOUNDRX, please change here!
#endif

#define SNDRX_MSKBIT                        SNDRX_BIT

#if SNDRX_CALIBRATE == 1
#define CALIBRATE_LED_PORT  PORTD
#define CALIBRATE_LED_DDR   DDRD
#define CALIBRATE_LED       6

#define CALIBRATE_LED_ON    CALIBRATE_LED_PORT |= 1<<CALIBRATE_LED
#define CALIBRATE_LED_OFF   CALIBRATE_LED_PORT &= ~(1<<CALIBRATE_LED)
#endif

/*-----------------------------------------------------------------------------------------------------------------------
 * Ringbuffer
 *-----------------------------------------------------------------------------------------------------------------------
 */
#if SNDRX_RINGBUFSIZE == 0
static volatile uint8_t                         inputch;
static volatile uint8_t                         inputch_stored;
#else
static volatile uint8_t                         ringbuf[SNDRX_RINGBUFSIZE];


static volatile uint8_t                         ringbuf_max;

static volatile uint8_t                         ringbuf_size;
static volatile uint8_t                         ringbuf_stop;
static volatile uint8_t                         ringbuf_start;
#endif

static volatile uint8_t                         sndrx_state = SNDRX_INFO_IDLE;

static volatile uint16_t                        sndrx_timeoutmsec;

/*-----------------------------------------------------------------------------------------------------------------------
 * LOGGING routines, ONLY for debugging!
 *-----------------------------------------------------------------------------------------------------------------------
 */
#if SNDRX_LOGGING == 1
#define LOGBUFSIZE                              250
static volatile uint8_t                         logcnt[LOGBUFSIZE];
static volatile uint8_t                         logbufidx = 0;

#define BAUD                                    38400L
#include <util/setbaud.h>

#ifdef UBRR0H

#define UART0_UBRRH                             UBRR0H
#define UART0_UBRRL                             UBRR0L
#define UART0_UCSRA                             UCSR0A
#define UART0_UCSRB                             UCSR0B
#define UART0_UCSRC                             UCSR0C
#define UART0_UDRE_BIT_VALUE                    (1<<UDRE0)
#define UART0_UCSZ1_BIT_VALUE                   (1<<UCSZ01)
#define UART0_UCSZ0_BIT_VALUE                   (1<<UCSZ00)
#ifdef URSEL0
#define UART0_URSEL_BIT_VALUE                   (1<<URSEL0)
#else
#define UART0_URSEL_BIT_VALUE                   (0)
#endif
#define UART0_TXEN_BIT_VALUE                    (1<<TXEN0)
#define UART0_RXEN_BIT_VALUE                    (1<<RXEN0)
#define UART0_UDR                               UDR0
#define UART0_U2X                               U2X0
#define UART0_RXC                               RXC0

#else

#define UART0_UBRRH                             UBRRH
#define UART0_UBRRL                             UBRRL
#define UART0_UCSRA                             UCSRA
#define UART0_UCSRB                             UCSRB
#define UART0_UCSRC                             UCSRC
#define UART0_UDRE_BIT_VALUE                    (1<<UDRE)
#define UART0_UCSZ1_BIT_VALUE                   (1<<UCSZ1)
#define UART0_UCSZ0_BIT_VALUE                   (1<<UCSZ0)
#ifdef URSEL
#define UART0_URSEL_BIT_VALUE                   (1<<URSEL)
#else
#define UART0_URSEL_BIT_VALUE                   (0)
#endif
#define UART0_TXEN_BIT_VALUE                    (1<<TXEN)
#define UART0_RXEN_BIT_VALUE                    (1<<RXEN)
#define UART0_UDR                               UDR
#define UART0_U2X                               U2X
#define UART0_RXC                               RXC

#endif

static void
sndrx_uart_init (void)
{
    UART0_UBRRH = UBRRH_VALUE;                                                                      // set baud rate
    UART0_UBRRL = UBRRL_VALUE;

#if USE_2X
    UART0_UCSRA |= (1<<UART0_U2X);
#else
    UART0_UCSRA &= ~(1<<UART0_U2X);
#endif

    UART0_UCSRC = UART0_UCSZ1_BIT_VALUE | UART0_UCSZ0_BIT_VALUE | UART0_URSEL_BIT_VALUE;
    UART0_UCSRB |= UART0_TXEN_BIT_VALUE;                                                            // enable UART TX
}

static void
sndrx_uart_putc (unsigned char ch)
{
    while (!(UART0_UCSRA & UART0_UDRE_BIT_VALUE))
    {
        ;
    }

    UART0_UDR = ch;
}

static void
sndrx_uart_puts (char * s)
{
    while (*s)
    {
        sndrx_uart_putc (*s++);
    }
}

SNDRX_STATIC void
sndrx_log (void)
{
    uint8_t i;
    int8_t  diff;
    char    buf[10];

    for (i = 0; i < LOGBUFSIZE; i++)
    {
        sndrx_uart_puts (itoa (i, buf, 10));
        sndrx_uart_putc (' ');
        sndrx_uart_puts (itoa (logcnt[i], buf, 10));
        sndrx_uart_putc (' ');

        if (logcnt[i] >= DATA_BIT_0_SCAN_TICKS_MIN && logcnt[i] <= DATA_BIT_0_SCAN_TICKS_MAX)
        {
            sndrx_uart_putc ('0');
            sndrx_uart_putc (' ');
            diff = (int8_t) logcnt[i] - DATA_BIT_0_SCAN_TICKS;
            sndrx_uart_puts (itoa (diff, buf, 10));
        }
        else if (logcnt[i] >= DATA_BIT_1_SCAN_TICKS_MIN && logcnt[i] <= DATA_BIT_1_SCAN_TICKS_MAX)
        {
            sndrx_uart_putc ('1');
            sndrx_uart_putc (' ');
            diff = (int8_t) logcnt[i] - DATA_BIT_1_SCAN_TICKS;
            sndrx_uart_puts (itoa (diff, buf, 10));
        }
        else if (logcnt[i] >= DATA_BLOCK_END_SCAN_TICKS_MIN && logcnt[i] <= DATA_BLOCK_END_SCAN_TICKS_MAX)
        {
            sndrx_uart_putc ('E');
        }
        else if (logcnt[i] >= DATA_BLOCK_START_SCAN_TICKS_MIN && logcnt[i] <= DATA_BLOCK_START_SCAN_TICKS_MAX)
        {
            sndrx_uart_putc ('S');
        }
        else
        {
            sndrx_uart_putc ('?');
        }
        sndrx_uart_puts ("\r\n");
    }
}

#endif // LOGGING == 1


/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  SNDRX init
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
SNDRX_STATIC void
sndrx_init (void)
{
    OCR1A   =  (F_CPU / PRESCALER / F_INTERRUPTS) - 1;                          // compare value: 1/1000 of CPU frequency
    TCCR1B  = (1 << WGM12) | (1 << CS11 | (1 << CS10));                         // switch CTC Mode on, set prescaler to 64

#ifdef TIMSK
    TIMSK  = 1 << OCIE1A;                                                       // OCIE1A: Interrupt by timer compare
#else
    TIMSK1  = 1 << OCIE1A;                                                      // OCIE1A: Interrupt by timer compare
#endif

    SNDRX_PORT |= 1<<SNDRX_BIT;                                                 // activate pullup
    SNDRX_DDR &= ~(1<<SNDRX_BIT);                                               // set pin to input

    SNDRX_MSK |= (1<<SNDRX_MSKBIT);                                             // Set PCINT mask

#ifdef PCICR
    PCICR |= (1<<SNDRX_PCIE);                                                   // enable PCINT (e.g. ATmega48/88/168/328)
#else
    GICR |= (1<<SNDRX_PCIE);                                                    // enable PCINT (e.g. ATmega162)
#endif

#if SNDRX_LOGGING == 1
    sndrx_uart_init ();
#endif

#if SNDRX_CALIBRATE == 1
    CALIBRATE_LED_OFF;
    CALIBRATE_LED_DDR |= (1<<CALIBRATE_LED);
#endif
}

static void
sndrx_set_error (uint8_t error)
{
    sndrx_state = error;
#if SNDRX_RINGBUFSIZE == 0
    inputch_stored = FALSE;
#else                   
    ringbuf_start = 0;
    ringbuf_stop = 0;
    ringbuf_size = 0;
#endif
}


/*-----------------------------------------------------------------------------------------------------------------------
 * Pin Change Interrupt
 *-----------------------------------------------------------------------------------------------------------------------
 */
ISR(SNDRX_PCINT_vect)
{
    static  uint8_t bit;
    static  uint8_t ch;
    static  uint8_t startbit_cnt;
    uint8_t         value;
    uint8_t         cnt;
    uint8_t         state;

    cnt = TCNT1;
    TCNT1 = 0;
    state = sndrx_state;

    value = input(SNDRX_PIN);

#if SNDRX_CALIBRATE == 1
    if (cnt < DATA_BIT_0_SCAN_TICKS + 3 && cnt >= DATA_BIT_0_SCAN_TICKS - 3)
    {
        CALIBRATE_LED_ON;
    }
    else
    {
        CALIBRATE_LED_OFF;
    }
#endif

#if SNDRX_LOGGING == 1
    if (logbufidx < LOGBUFSIZE)
    {
        logcnt[logbufidx++] = cnt;
    }
#endif

    if (cnt >= DATA_BLOCK_START_SCAN_TICKS_MIN && cnt <= DATA_BLOCK_START_SCAN_TICKS_MAX)                       // start bit
    {
        startbit_cnt++;

        if (startbit_cnt == 2)
        {
            bit = 0;
            state = SNDRX_INFO_TRANSMIT;
            ch = 0;
        }
    }
    else
    {
        startbit_cnt = 0;

        if (state == SNDRX_INFO_TRANSMIT)
        {
            if (cnt >= DATA_BIT_0_SCAN_TICKS_MIN && cnt <= DATA_BIT_0_SCAN_TICKS_MAX)                           // 0-bit
            {
                ch <<= 1;
            }
            else if (cnt >= DATA_BIT_1_SCAN_TICKS_MIN && cnt <= DATA_BIT_1_SCAN_TICKS_MAX)                      // 0-bit
            {
                ch <<= 1;
                ch |= 1;
            }
            else if (cnt >= DATA_BLOCK_END_SCAN_TICKS_MIN && cnt <= DATA_BLOCK_END_SCAN_TICKS_MAX && bit == 0)  // stop bit
            {
                state = SNDRX_INFO_IDLE;
            }
            else                                                                                                // frame error
            {
                sndrx_set_error (SNDRX_ERROR_FRAME);
            }

            if (state == SNDRX_INFO_TRANSMIT)
            {
                bit++;

                if (bit == 8)
                {
#if SNDRX_RINGBUFSIZE == 0
                    if (inputch_stored)
                    {
                        sndrx_set_error (SNDRX_ERROR_OVERFLOW);
                    }
                    else
                    {
                        inputch = ch;
                        inputch_stored = TRUE;
                    }
#else
                    if (ringbuf_size < SNDRX_RINGBUFSIZE)
                    {
                        uint8_t stop = ringbuf_stop;

                        ringbuf[stop++] = ch;

                        if (stop >= SNDRX_RINGBUFSIZE)                      // at end of ringbuffer?
                        {                                                   // yes
                            stop = 0;                                       // reset to beginning
                        }

                        ringbuf_stop = stop;
                        ringbuf_size++;
                        if (ringbuf_size > ringbuf_max)
                        {
                            ringbuf_max = ringbuf_size;
                        }
                    }
                    else
                    {
                        sndrx_set_error (SNDRX_ERROR_OVERFLOW);
                    }
#endif

                    bit = 0;
                    ch = 0;
                }
            }
        }
    }

    sndrx_state = state;
}


/*-----------------------------------------------------------------------------------------------------------------------
 * Timer Interrupt, called every 1 msec
 *-----------------------------------------------------------------------------------------------------------------------
 */
ISR(TIMER1_COMPA_vect)
{
    uint16_t        timeoutmsec = sndrx_timeoutmsec;

    if (sndrx_state == SNDRX_INFO_TRANSMIT)
    {
        sndrx_set_error (SNDRX_ERROR_TIMEOUT);
    }
    else
    {
        if (timeoutmsec)
        {
            sndrx_timeoutmsec = timeoutmsec - 1;
        }
    }
}


/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  SNDRX poll: poll with timeout (msec)
 *
 *  Return value    Remark                                          sndrx_status() will then return:
 *
 *      0           no character available within timeout           SNDRX_INFO_IDLE
 *      1           one character available, stored in (*chp)       SNDRX_INFO_TRANSMIT
 *     -1           error occured                                   SNDRX_ERROR_xxx
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
SNDRX_STATIC int8_t
sndrx_poll (uint8_t * chp, uint16_t timeoutmsec)
{
    int8_t          rtc;

    if (sndrx_state >= SNDRX_ERROR_FRAME)
    {
        rtc = -1;
    }
    else
    {
        if (timeoutmsec > 0)
        {
            cli ();
            sndrx_timeoutmsec = timeoutmsec;
            sei ();

#if SNDRX_RINGBUFSIZE == 0
            while (sndrx_timeoutmsec && ! inputch_stored)
#else
            while (sndrx_timeoutmsec && ! ringbuf_size)
#endif
            {
                ;
            }
        }

#if SNDRX_RINGBUFSIZE == 0
        if (inputch_stored)
        {
            rtc = 1;

            *chp = inputch;                             // get character, increment offset
            cli ();
            inputch_stored = FALSE;
            sei ();
        }
#else
        if (ringbuf_size)
        {
            uint8_t start;

            rtc = 1;

            cli ();
            start = ringbuf_start;
            *chp = ringbuf[start++];                    // get character, increment offset

            if (start == SNDRX_RINGBUFSIZE)             // at end of ringbuffer?
            {                                           // yes
                start = 0;                              // reset to beginning
            }

            ringbuf_start = start;
            ringbuf_size--;                             // decrement size
            sei ();
        }
#endif
        else
        {
            rtc = 0;
        }
    }
    return rtc;
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  SNDRX status
 *
 *  Return values:
 *
 *    SNDRX_INFO_IDLE                         INFO:  sndrx is idle
 *    SNDRX_INFO_TRANSMIT                     INFO:  sender is transmitting
 *    SNDRX_ERROR_FRAME                       ERROR: frame error occured
 *    SNDRX_ERROR_OVERFLOW                    ERROR: ringbuffer overflow
 *    SNDRX_ERROR_TIMEOUT                     ERROR: timeout occured (buffer underrun)
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  SNDRX stop transmission
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
SNDRX_STATIC void
sndrx_stop (void)
{
    cli ();
    sndrx_set_error (SNDRX_INFO_IDLE);
    sei ();
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  SNDRX done: deactivate timer used by sndrx - only relevant if bootloader
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
SNDRX_STATIC void
sndrx_done (void)
{
    TIMSK1  = 0;                                                                // reset OCIE1A
    TCCR1B  = 0;                                                                // reset TCCR1B
    OCR1A   = 0;                                                                // reset OCR1A

#ifdef PCICR
    PCICR &= ~(1<<SNDRX_PCIE);                                                  // disable PCINT (e.g. ATmega48/88/168/328)
#else
    GICR &= ~(1<<SNDRX_PCIE);                                                   // disable PCINT (e.g. ATmega162)
#endif
    SNDRX_MSK &= ~(1<<SNDRX_MSKBIT);                                            // Reset PCINT mask

    SNDRX_PORT &= ~(1<<SNDRX_BIT);                                              // deactivate pullup
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  SNDRX status
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#ifndef SNDRX_BOOTLOADER
SNDRX_STATIC uint8_t
sndrx_status (void)
{
    uint8_t rtc = sndrx_state;

    if (sndrx_state >= SNDRX_ERROR_FRAME)
    {
        cli ();
        sndrx_state = SNDRX_INFO_IDLE;
        sei ();
    }
    return rtc;
}
#endif
