/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * @file sndflash.c
 *
 * Copyright (c) 2011 Robert & Frank Meyer - frank(at)fli4l.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "sndtx.h"

#define TRUE        1
#define FALSE       0

#define F_SAMPLES           41667

#define SPM_PAGESIZE        128
#define LINES_PER_PAGE      8

typedef                     unsigned char uint8_t;

static unsigned char        flash[65536];
static unsigned int         address_low     = 0xffff;
static unsigned int         address_high    = 0x0000;

/*------------------------------------------------------------------------------------------------------------------------
 * hex2toi ()
 *------------------------------------------------------------------------------------------------------------------------
 */
static uint8_t
hex2toi (char * p)
{
    uint8_t i = 0;
    uint8_t ch;
    uint8_t idx;

    for (idx = 0; idx < 2; idx++)
    {
        ch = *p++;

        if (ch >= '0' && ch <= '9')
        {
            i |= (ch - '0');
        }
        else if (ch >= 'A' && ch <= 'F')
        {
            i |= (ch - 'A' + 10);
        }
        else if (*p >= 'a' && *p <= 'f')
        {
            i |= (ch - 'a' + 10);
        }

        if (idx < 1)
        {
            i <<= 4;
        }
    }

    return (i);
}

/*------------------------------------------------------------------------------------------------------------------------
 * check_hex_file ()
 *------------------------------------------------------------------------------------------------------------------------
 */
int
check_hex_file (FILE * fp)
{
    char            buf[64];
    int             line_ok;
    unsigned char   len;
    unsigned char   ch;
    unsigned char   idx;
    unsigned char   datalen;
    unsigned int    address;
    unsigned char   addrh;
    unsigned char   addrl;
    unsigned char   rectype;
    char *          dataptr;
    unsigned char   chcksum;
    int             sum = 0;

    while (fgets (buf, 64, fp))
    {
        line_ok = FALSE;

        len = strlen (buf);

        if (len > 2 && *(buf + len - 2) == '\r' && *(buf + len - 1) == '\n')
        {
            *(buf + len - 2) = '\0';
        }

        len = strlen (buf);

        if (buf[0] == ':' && len >= 11)
        {
            datalen = hex2toi (buf +  1);
            addrh   = hex2toi (buf +  3);
            addrl   = hex2toi (buf +  5);
            address = addrh << 8 | addrl;
            rectype = hex2toi (buf + 7);
            dataptr = buf + 9;

            sum = datalen + addrh + addrl + rectype;

            if (len == 9 + 2 * datalen + 2)
            {
                for (idx = 0; idx < datalen; idx++)
                {
                    ch = hex2toi (dataptr);

                    flash[address + idx] = ch;

                    if (address_low > address + idx)
                    {
                        address_low = address + idx;
                    }

                    if (address_high < address + idx)
                    {
                        address_high = address + idx;
                    }

                    sum += ch;
                    dataptr +=2;
                }

                sum = (0x100 - (sum & 0xff)) & 0xff;
                chcksum = hex2toi (dataptr);

                if (sum == chcksum)
                {
                    line_ok = TRUE;
                }
                else
                {
                    fprintf (stderr, "invalid checksum: sum: 0x%02x chcksum: 0x%02x\n", sum, chcksum);
                }
            }
            else
            {
                fprintf (stderr, "invalid len: %d, expected %d\n", len, 12 + 2 * datalen + 2);
            }
        }
        else
        {
            fprintf (stderr, "invalid INTEL HEX format, len: %d\n", len);
        }

        if (! line_ok)
        {
            fprintf (stderr, "line '%s' not ok\n", buf);
            return (ERR);
        }
    }

    return (OK);
}


/*------------------------------------------------------------------------------------------------------------------------
 * usage ()
 *------------------------------------------------------------------------------------------------------------------------
 */
static void
usage (char * pgm_name)
{
    fprintf (stderr, "usage: %s [-s samples] file [outputfile.wav]\n", pgm_name);
    exit (1);
}

/*------------------------------------------------------------------------------------------------------------------------
 * main ()
 *------------------------------------------------------------------------------------------------------------------------
 */
int
main (int argc, char ** argv)
{
    char *          pgmname;
    char *          file;
    char *          target = (char *) NULL;
    FILE *          fp;
    int             rtc = 0;
    int             offset = 0;
    int             samples;
    long            bufsize;
    int             buflen = 0;
    int             page_low;
    int             page_high;
    char *          bufp;
    long            flashsize;
    int             sum = 0;

    pgmname = argv[0];

    samples = F_SAMPLES;

    pgmname = argv[0];

    while (argc > 1 && *argv[1] == '-')
    {
        if (argc > 2 && ! strcmp (argv[1], "-s"))
        {
            argc--;
            argv++;
            samples = atoi (argv[1]);

            if (samples < 8000)
            {
                fprintf (stderr, "%s: invalid value of samples: %s, minimum is 8000.\n", pgmname, argv[1]);
                exit (1);
            }

            if (samples > 44100)
            {
                fprintf (stderr, "%s: invalid value of samples: %s, maximum is 44100.\n", pgmname, argv[1]);
                exit (1);
            }
            argc--;
            argv++;
        }
        else
        {
            usage (pgmname);
        }
    }

    if (argc == 3)
    {
        file = argv[1];
        target = argv[2];
    }
    else if (argc == 2)
    {
        file = argv[1];
    }
    else
    {
        usage (pgmname);
    }

    fp = fopen (file, "rb");

    if (fp)
    {
        if (check_hex_file (fp) == OK)
        {
            if (address_high - address_low > 0)
            {
                int             idx;
                int             pgcnt;
                int             n_pages;
                unsigned char * s;
                unsigned char * t;
                int             chksum;

                page_low        = address_low / SPM_PAGESIZE;
                page_high       = address_high / SPM_PAGESIZE + 1;

                address_low     = page_low * SPM_PAGESIZE;
                address_high    = page_high * SPM_PAGESIZE;
                flashsize       = address_high - address_low;
                n_pages         = flashsize / SPM_PAGESIZE;
                bufsize         = 1 + 4 + flashsize + n_pages;
                
                printf ("low = 0x%04x high = 0x%04x flash size = 0x%04x\n", address_low, address_high, flashsize);

                bufp = calloc (1, bufsize);

                if (! bufp)
                {
                    perror ("calloc");
                    exit (1);
                }

                bufp[0] = '$';
                bufp[1] = (address_low & 0xFF00) >> 8;
                bufp[2] = address_low & 0x00FF;

#if 1
                bufp[3] = (n_pages & 0xFF00) >> 8;
                bufp[4] = n_pages & 0x00FF;
#else
                bufp[3] = n_pages;
#endif

                printf ("!%d!\n", bufp[4]);
                t = bufp + 1 + 4;
                s = flash + address_low;

#if 1
                for (pgcnt = 0; pgcnt < n_pages; pgcnt++)
                {
                    chksum = 0;

                    for (idx = 0; idx < SPM_PAGESIZE; idx++)
                    {
                        *t++ = *s;
                        chksum ^= *s;
                        s++;
                    }
                    *t++ = chksum;
                }
#else
                memcpy (t, s, flashsize);
#endif
                if (sound_play (bufp, bufsize, samples, target) == ERR)
                {
                    rtc = 1; 
                } 
            }
        }

        fclose (fp);
    }
    else
    {
        perror (file);
    }

    return rtc;
}
