/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * @file main.c
 *
 * THIS IS ONLY A TEST/EXAMPLE MAIN FILE FOR SNDRX!
 *
 * This example program echoes received characters via SOUNDRX on the UART by 38400 baud.
 *
 *   Green LED shows TRANSMIT status
 *   Red LED shows ERROR status
 *
 * Copyright (c) 2011 Robert Meyer, Frank Meyer - frank(at)fli4l.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

#include <stdlib.h>
#include <inttypes.h>
#include <avr/io.h>
#include <util/delay.h>
#include <avr/interrupt.h>
#include <avr/wdt.h>


#include "sndrxconfig.h"
#include "sndrx.h"


#define FALSE   0
#define TRUE    1

/*-----------------------------------------------------------------------------------------------------------------------
 * LEDs: CHANGE HERE!
 *
 *   Green LED shows TRANSMIT status
 *   Red LED shows ERROR status
 *-----------------------------------------------------------------------------------------------------------------------
 */
#define LED_GREEN_PORT      PORTD
#define LED_GREEN_DDR       DDRD
#define LED_GREEN           6

#define LED_RED_PORT        PORTB
#define LED_RED_DDR         DDRB
#define LED_RED             0

#define LED_GREEN_ON        LED_GREEN_PORT |= 1<<LED_GREEN
#define LED_GREEN_OFF       LED_GREEN_PORT &= ~(1<<LED_GREEN)
#define LED_RED_ON          LED_RED_PORT |= 1<<LED_RED
#define LED_RED_OFF         LED_RED_PORT &= ~(1<<LED_RED)

/*-----------------------------------------------------------------------------------------------------------------------
 * Initialize LEDs
 *-----------------------------------------------------------------------------------------------------------------------
 */
static void
led_init (void)
{
    LED_GREEN_OFF;
    LED_GREEN_DDR |= (1<<LED_GREEN);

    LED_RED_OFF;
    LED_RED_DDR |= (1<<LED_RED);
}

/*-----------------------------------------------------------------------------------------------------------------------
 * UART routines
 *-----------------------------------------------------------------------------------------------------------------------
 */
#define BAUD                                    38400L
#include <util/setbaud.h>

#ifdef UBRR0H

#define UART0_UBRRH                             UBRR0H
#define UART0_UBRRL                             UBRR0L
#define UART0_UCSRA                             UCSR0A
#define UART0_UCSRB                             UCSR0B
#define UART0_UCSRC                             UCSR0C
#define UART0_UDRE_BIT_VALUE                    (1<<UDRE0)
#define UART0_UCSZ1_BIT_VALUE                   (1<<UCSZ01)
#define UART0_UCSZ0_BIT_VALUE                   (1<<UCSZ00)
#ifdef URSEL0
#define UART0_URSEL_BIT_VALUE                   (1<<URSEL0)
#else
#define UART0_URSEL_BIT_VALUE                   (0)
#endif
#define UART0_TXEN_BIT_VALUE                    (1<<TXEN0)
#define UART0_RXEN_BIT_VALUE                    (1<<RXEN0)
#define UART0_UDR                               UDR0
#define UART0_U2X                               U2X0
#define UART0_RXC                               RXC0

#else

#define UART0_UBRRH                             UBRRH
#define UART0_UBRRL                             UBRRL
#define UART0_UCSRA                             UCSRA
#define UART0_UCSRB                             UCSRB
#define UART0_UCSRC                             UCSRC
#define UART0_UDRE_BIT_VALUE                    (1<<UDRE)
#define UART0_UCSZ1_BIT_VALUE                   (1<<UCSZ1)
#define UART0_UCSZ0_BIT_VALUE                   (1<<UCSZ0)
#ifdef URSEL
#define UART0_URSEL_BIT_VALUE                   (1<<URSEL)
#else
#define UART0_URSEL_BIT_VALUE                   (0)
#endif
#define UART0_TXEN_BIT_VALUE                    (1<<TXEN)
#define UART0_RXEN_BIT_VALUE                    (1<<RXEN)
#define UART0_UDR                               UDR
#define UART0_U2X                               U2X
#define UART0_RXC                               RXC

#endif

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Initialize  UART
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void
uart_init (void)
{
    UART0_UBRRH = UBRRH_VALUE;                                                                      // set baud rate
    UART0_UBRRL = UBRRL_VALUE;

#if USE_2X
    UART0_UCSRA |= (1<<UART0_U2X);
#else
    UART0_UCSRA &= ~(1<<UART0_U2X);
#endif

    UART0_UCSRC = UART0_UCSZ1_BIT_VALUE | UART0_UCSZ0_BIT_VALUE | UART0_URSEL_BIT_VALUE;
    UART0_UCSRB |= UART0_TXEN_BIT_VALUE;                                                            // enable UART TX
}


/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Send character
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void
uart_putc (unsigned char ch)
{
    while (!(UART0_UCSRA & UART0_UDRE_BIT_VALUE))
    {
        ;
    }

    UART0_UDR = ch;
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Send string
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void
uart_puts (char * s)
{
    while (*s)
    {
        uart_putc (*s++);
    }
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Main function
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
int
main(void)
{
#if SNDRX_CALIBRATE == 0
    uint8_t ch;
    int8_t rtc;
#endif

    wdt_disable ();                                                     // disable watchdog
    sndrx_init();                                                       // initialize SOUNDRX
    uart_init();                                                        // initialize UART
    led_init ();                                                        // initialize LED ports
    sei ();                                                             // enable interrupts

    uart_puts ("Hello, this is sndrx.\r\n");                            // show welcome message

#if SNDRX_LOGGING == 1                                                  // only for debugging, here not used
    _delay_ms (5000);
    sndrx_log ();
#endif

    while(1)
    {
#if SNDRX_CALIBRATE == 0
        rtc = sndrx_poll (&ch, 200);                                    // read a character from SOUNDRX

        if (rtc > 0)                                                    // charater read
        {
            uart_putc (ch);                                             // echo it on UART
        }

        switch (sndrx_status ())                                        // get current status
        {
            case SNDRX_INFO_IDLE:                                       // SNDRX is idle
            {
                LED_GREEN_OFF;                                          // switch TRANSMIT LED OFF
                break;
            }
            case SNDRX_INFO_TRANSMIT:                                   // SNDRX receives data
            {
                LED_GREEN_ON;                                           // switch TRANSMIT LED ON
                LED_RED_OFF;                                            // switch ERROR LED OFF
                break;
            }
            case SNDRX_ERROR_FRAME:                                     // SNDRX got a frame error
            {
                LED_GREEN_OFF;                                          // switch TRANSMIT LED OFF
                LED_RED_ON;                                             // switch ERROR LED ON
                uart_puts ("\r\nframe error!\r\n");
                break;
            }
            case SNDRX_ERROR_OVERFLOW:                                  // SNDRX detected an overflow error
            {
                LED_GREEN_OFF;                                          // switch TRANSMIT LED OFF
                LED_RED_ON;                                             // switch ERROR LED ON
                uart_puts ("\r\noverflow error!\r\n");
                break;
            }
            case SNDRX_ERROR_TIMEOUT:                                   // SNDRX detected a buffer underrun/timeout error
            {
                LED_GREEN_OFF;                                          // switch TRANSMIT LED OFF
                LED_RED_ON;                                             // switch ERROR LED ON
                uart_puts ("\r\ntimeout error!\r\n");
                break;
            }
        }
#endif
    }
    return 0;
}
