/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * @file sndtx.c
 *
 * Copyright (c) 2011 Robert & Frank Meyer - frank(at)fli4l.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <windows.h>
#include <mmsystem.h>

#include "sndtx.h"

#pragma comment (lib, "C:\\Program Files\\Microsoft SDKs\\Windows\\v6.0A\\Lib\\WinMM.lib")
typedef unsigned char           uint8_t;

#define TRUE                    1
#define FALSE                   0

#define TRAILER_LEN             0

#define DATA_BIT_0_TICKS        2
#define DATA_BIT_1_TICKS        3
#define DATA_BLOCK_END_TICKS    6
#define DATA_BLOCK_START_TICKS  9

static volatile int             sound_ready = FALSE;

static void CALLBACK
mycallback (HWAVEOUT hWaveOut, UINT uMsg, DWORD dwInstance, DWORD dwParam1, DWORD dwParam2)
{
    // printf ("mycallback\n");
    if (uMsg == WOM_DONE)
    {
        // printf ("done\n");
        sound_ready = TRUE;
    }
    else if (uMsg == WOM_OPEN)
    {
        // printf ("open\n");
    }
    else if (uMsg == WOM_CLOSE)
    {
        // printf ("close\n");
    }
}

static BYTE *
convert_ticks_to_wave (BYTE * bufp, int ticks)
{
    static int  last_value = 0;
    int         value;
    int         idx;

    value = last_value ? 0 : 255;

    for (idx = 0; idx < ticks; idx++)
    {
        *bufp++ = value;
    }

    last_value = value;
    return bufp;
}

static BYTE *
convert_bit_to_wave (BYTE * bufp, int bit)
{
    if (bit)
    {
        bufp = convert_ticks_to_wave (bufp, DATA_BIT_1_TICKS);
    }
    else
    {
        bufp = convert_ticks_to_wave (bufp, DATA_BIT_0_TICKS);
    }
    return bufp;
}

static int
calc_byte_to_wave (int ch)
{
    int idx;
    int bit;
    int sum = 0;

    for (idx = 7; idx >= 0; idx--)
    {
        bit = ch & (1<<idx);

        if (bit)
        {
            sum += DATA_BIT_1_TICKS;
        }
        else
        {
            sum += DATA_BIT_0_TICKS;
        }
    }
    return sum;
}

static BYTE *
convert_block_start_to_wave (BYTE * bufp)
{
    bufp = convert_ticks_to_wave (bufp, DATA_BLOCK_START_TICKS);
    bufp = convert_ticks_to_wave (bufp, DATA_BLOCK_START_TICKS);
    return bufp;
}

static BYTE *
convert_block_end_to_wave (BYTE * bufp)
{
    bufp = convert_ticks_to_wave (bufp, DATA_BLOCK_END_TICKS);
    bufp = convert_ticks_to_wave (bufp, DATA_BLOCK_END_TICKS);
    return bufp;
}

static BYTE *
convert_byte_to_wave (BYTE * bufp, int ch)
{
    int idx;
    int bit;

    for (idx = 7; idx >= 0; idx--)
    {
        bit = ch & (1<<idx);

        bufp = convert_bit_to_wave (bufp, bit);
    }

    return bufp;
}


static long
calc_buffersize (unsigned char * buf, long bufsize)
{
    unsigned char * bufp;
    long            size = 0;

    size += calc_byte_to_wave (0x00);
    size += calc_byte_to_wave (0x00);

    size += 2 * DATA_BLOCK_START_TICKS;

    for (bufp = buf; bufsize > 0; bufsize--, bufp++)
    {
        size += calc_byte_to_wave (*bufp);
    }

    size += 2 * DATA_BLOCK_END_TICKS;

    size += TRAILER_LEN * calc_byte_to_wave (0x00);

    return size;
}

static BYTE *
fill_buffer (unsigned char * buf, long bufsize, long wavesize)
{
    int             idx;
    BYTE *          wavebuf;
    BYTE *          wavebufp;
    unsigned char * bufp;

    wavebuf = malloc (wavesize);

    if (wavebuf)
    {
        wavebufp = wavebuf;

        wavebufp = convert_byte_to_wave (wavebufp, 0x00);
        wavebufp = convert_byte_to_wave (wavebufp, 0x00);

        wavebufp = convert_block_start_to_wave (wavebufp);

        for (bufp = buf; bufsize > 0; bufsize--, bufp++)
        {
            wavebufp = convert_byte_to_wave (wavebufp, *bufp);
        }

        wavebufp = convert_block_end_to_wave (wavebufp);

        for (idx = 0; idx < TRAILER_LEN; idx++)
        {
            wavebufp = convert_byte_to_wave (wavebufp, 0x00);
        }
    }
    return (wavebuf);
}

typedef struct
{
    char        ID[4];
    int         ChunkSize;
    char        Format[4];
} RIFF_HEADER;

typedef struct
{
    WORD        AudioFormat;
    WORD        NumChannels;
    int         SampleRate;
    int         ByteRate;
    WORD        BlockAlign;
    WORD        BitsPerSample;
} FMT_DATA;

typedef struct
{
    char        FmtID[4];
    int         FmtSize;
    FMT_DATA    FmtData;
} FMT_HEADER;

typedef struct
{
    char        DataID[4];
    int         DataSize;
} DATA_HEADER;

static int
waveSave (char * target, WAVEFORMATEX * waveFormatp, BYTE * wavebuffer, long wavesize)
{
    FILE *          file;
    RIFF_HEADER     RiffHeader;
    FMT_HEADER      FmtHeader;
    DATA_HEADER     DataHeader;
    int             rtc;

    //------------------------------------------------------------------------------------------------------------------
    // RIFF Header
    //------------------------------------------------------------------------------------------------------------------
    RiffHeader.ID[0]        = 'R';
    RiffHeader.ID[1]        = 'I';
    RiffHeader.ID[2]        = 'F';
    RiffHeader.ID[3]        = 'F';

    RiffHeader.ChunkSize    = sizeof (RIFF_HEADER) + sizeof (FMT_HEADER) + wavesize;        // 36 + wavesize
                                                                                            // == complete file size - 8
    if (RiffHeader.ChunkSize != 36 + wavesize)
    {
        fprintf (stderr, "Internal error 1. EXIT.\n");
        exit (1);
    }

    RiffHeader.Format[0]    = 'W';
    RiffHeader.Format[1]    = 'A';
    RiffHeader.Format[2]    = 'V';
    RiffHeader.Format[3]    = 'E';

    //------------------------------------------------------------------------------------------------------------------
    // FMT Header
    //------------------------------------------------------------------------------------------------------------------
    FmtHeader.FmtID[0]      = 'f';
    FmtHeader.FmtID[1]      = 'm';
    FmtHeader.FmtID[2]      = 't';
    FmtHeader.FmtID[3]      = ' ';

    FmtHeader.FmtSize       = sizeof (FMT_DATA);                                                    // 16

    if (FmtHeader.FmtSize != 16)
    {
        fprintf (stderr, "Internal error 2. EXIT.\n");
        exit (1);
    }

    FmtHeader.FmtData.AudioFormat       = waveFormatp->wFormatTag;                                  // PCM
    FmtHeader.FmtData.NumChannels       = waveFormatp->nChannels;                                   // MONO
    FmtHeader.FmtData.SampleRate        = waveFormatp->nSamplesPerSec;                              // 12000
    FmtHeader.FmtData.ByteRate          = waveFormatp->nSamplesPerSec * waveFormatp->nBlockAlign;   // 12000
    FmtHeader.FmtData.BlockAlign        = waveFormatp->nBlockAlign;                                 // 1
    FmtHeader.FmtData.BitsPerSample     = waveFormatp->wBitsPerSample;                              // 8

    //------------------------------------------------------------------------------------------------------------------
    // DATA Header
    //------------------------------------------------------------------------------------------------------------------
    DataHeader.DataID[0]    = 'd';
    DataHeader.DataID[1]    = 'a';
    DataHeader.DataID[2]    = 't';
    DataHeader.DataID[3]    = 'a';
    DataHeader.DataSize     = wavesize; 

    file = fopen (target, "wb");

    if (file)
    {
        fwrite (&RiffHeader, sizeof (RiffHeader), 1, file);                                         // 12 bytes
        fwrite (&FmtHeader,  sizeof (FmtHeader),  1, file);                                         // 24 bytes (12 + 24 = 36)
        fwrite (&DataHeader, sizeof (DataHeader), 1, file);                                         //  8 bytes
        fwrite (wavebuffer, wavesize, 1, file);                                                     // data
        fclose (file); 
        rtc = OK;
    }
    else
    {
        perror (target);
        rtc = ERR;
    }
    return rtc;
}

int
sound_play (unsigned char * buf, long bufsize, int samples, char * target)
{
    static HWAVEOUT         outHandle;
    static WAVEFORMATEX     waveFormat;
    static WAVEHDR          hdr;
    unsigned long           result;
    BYTE *                  wavebuf;
    long                    wavesize;
    int                     rtc = ERR;

    ZeroMemory (&waveFormat, sizeof(WAVEFORMATEX));
    ZeroMemory (&hdr, sizeof(WAVEHDR));

    wavesize = calc_buffersize (buf, bufsize);
    wavebuf = fill_buffer (buf, bufsize, wavesize);

    if (wavebuf)
    {
        hdr.lpData          = (LPSTR) wavebuf;
        hdr.dwBufferLength  = wavesize;
        hdr.dwBytesRecorded = 0;
        hdr.dwUser          = 0;
        hdr.dwFlags         = 0;
        hdr.dwLoops         = 0;
        hdr.lpNext          = 0;
        hdr.reserved        = 0;

        /* Initialize the WAVEFORMATEX for 8-bit, xx kHz, mono */
        waveFormat.wFormatTag       = WAVE_FORMAT_PCM;
        waveFormat.nChannels        = 1;
        waveFormat.nSamplesPerSec   = samples;
        waveFormat.wBitsPerSample   = 8;
        waveFormat.nBlockAlign      = waveFormat.nChannels * (waveFormat.wBitsPerSample / 8);
        waveFormat.nAvgBytesPerSec  = waveFormat.nSamplesPerSec * waveFormat.nBlockAlign;
        waveFormat.cbSize           = 0;

        if (target)
        {
            rtc = waveSave (target, &waveFormat, wavebuf, wavesize);
        }
        else
        {
            result = waveOutOpen (&outHandle, WAVE_MAPPER, &waveFormat, (DWORD_PTR) mycallback, 0, CALLBACK_FUNCTION);

            if (result == MMSYSERR_NOERROR)
            {
                if (waveOutPrepareHeader (outHandle, &hdr, sizeof (WAVEHDR)) == MMSYSERR_NOERROR)
                {
                    if (waveOutWrite (outHandle, &hdr, sizeof (WAVEHDR)) == MMSYSERR_NOERROR)
                    {
                        int sec;
                        int maxsec = wavesize / waveFormat.nSamplesPerSec;

                        printf ("buffer size: %ld wave size: %d time: %02d:%02d speed: %d bytes/sec\n",
                                bufsize, wavesize, maxsec / 60, maxsec % 60, bufsize / (maxsec ? maxsec : 1));

                        rtc = OK;

                        for (sec = 0; ; sec++)
                        {
                            if (sound_ready)
                            {
                                break;
                            }
                            printf ("\rTIME: %02d:%02d ETC: %02d:%02d %3d%%",
                                    sec / 60, sec % 60, (maxsec - sec) / 60, (maxsec - sec) % 60,
                                    maxsec ? ((sec * 100) / maxsec) : 100, maxsec);
                            fflush (stdout);
                            Sleep (1000);
                        }
                        putchar ('\n');
                    }
                    else
                    {
                        fprintf(stderr, "error writing wave data!\r\n");
                    }
                }
                else
                {
                    fprintf(stderr, "error preparing wave header!\r\n");
                }

                waveOutClose (outHandle);
            }
            else
            {
                fprintf(stderr, "error opening the preferred Digital Audio Out device!\r\n");
            }
        }
    }
    else
    {
        perror ("malloc");
    }

    return rtc;
}
