//-----------------------------------------------------------------------------
// INCLUDES
//-----------------------------------------------------------------------------
#include <avr/io.h>
#include <util/delay.h>

#include "timer.h"
#include "adc.h"
#include "led.h"
#include "digitalIo.h"


//-----------------------------------------------------------------------------
// DEFINES
//-----------------------------------------------------------------------------
#define LED_DDR             DDRC
#define LED_PORT            PORTC
#define LED_PINNR           2

#define AD_PULL_DOWN_DDR    DDRC
#define AD_PULL_DOWN_PORT   PORTC
#define AD_PULL_DOWN_PINNR  1

#define AD_CHANNEL_NR       0


//-----------------------------------------------------------------------------
// CONSTANTS
//-----------------------------------------------------------------------------
const uint16_t  MinimalVoltage  = 6000;    // [mV]

const uint16_t  LedOnTime       = 100;      // [ms]
const uint8_t   LedOffTime      = 15;       // [s]


//-----------------------------------------------------------------------------
// LOCAL FUNCTION DEKLARATIONS
//-----------------------------------------------------------------------------
static void     pullDownActive      ( void );
static void     pullDownInactive    ( void );


//-----------------------------------------------------------------------------
// VARIABELS
//-----------------------------------------------------------------------------
uint8_t  ledSecCnt;


//-----------------------------------------------------------------------------
void ledControlTick ( const Settings *setting ) {

  if (doSleep == FALSE) {
    ledOn();
  } else {
  
    if (setting->flashLed == TRUE) {
      if (ledSecCnt == 0) {
        ledSecCnt = LedOffTime;
      } else if (ledSecCnt == 1) {
        ledOn();
        _delay_ms(LedOnTime);
        ledOff();
      }
      ledSecCnt--;
    } else {
      ledOff();
      ledSecCnt = 1;
    }
  }
}

//-----------------------------------------------------------------------------
void voltageControlTick ( Settings *setting ) {
  uint16_t  adc;
  uint32_t  spannung;
  
  pullDownActive();
  _delay_ms(100);
  adc = adc_readChannel(AD_CHANNEL_NR);
  pullDownInactive();
  
  // Uadc = Uref /1024 * adc       Uref = 1100 mV
  // Ubat = Uadc / 10 * 110 = Uref * adc * 11 / 1024
  
  // Anzahl Bits = ln(1100 * 1024 * 11) / ln(2) = 23.6 --> uint32_t
  spannung = (uint32_t)adc * 1100 * 11;
  spannung >>= 10; // /= 1024
  
  setting->voltage = (uint16_t) spannung;
  
  setting->flashLed = (setting->voltage < MinimalVoltage) ? TRUE : FALSE;
}

//-----------------------------------------------------------------------------
void ledOn ( void ) {
  LED_PORT &= ~(1 << LED_PINNR);  // low active
}

//-----------------------------------------------------------------------------
void ledOff ( void ) {
  LED_PORT |= (1 << LED_PINNR);   // low active
}

//-----------------------------------------------------------------------------
void initLed ( void ) {
  LED_DDR |= (1 << LED_PINNR);
  ledOff();
}


//-----------------------------------------------------------------------------
// LOCAL FUNCTIONS
//-----------------------------------------------------------------------------
/// \brief Output low
static void pullDownActive ( void ) {
  AD_PULL_DOWN_PORT &= ~(1 << AD_PULL_DOWN_PINNR);
  AD_PULL_DOWN_DDR |= (1 << AD_PULL_DOWN_PINNR);
}

//-----------------------------------------------------------------------------
/// \brief Input, no PullUp
static void pullDownInactive ( void ) {
  AD_PULL_DOWN_PORT &= ~(1 << AD_PULL_DOWN_PINNR);
  AD_PULL_DOWN_DDR &= ~(1 << AD_PULL_DOWN_PINNR);
}
