//-----------------------------------------------------------------------------
// INCLUDES
//-----------------------------------------------------------------------------
#include <avr/io.h>
#include <avr/power.h>

#include "uart.h"


//-----------------------------------------------------------------------------
void USART_Init ( uint32_t baud ) {
  //Set baud rate
  UBRR0H = (uint8_t)((F_CPU/16/baud-1)>>8);
  UBRR0L = (uint8_t)(F_CPU/16/baud-1);
  // Enable receiver and transmitter
  UCSR0B = (1<<RXEN0)|(1<<TXEN0);
  // Set frame format: 8data, 2stop bit
  UCSR0C = (1<<USBS0)|(3<<UCSZ00);
}

//-----------------------------------------------------------------------------
void USART_ReInit ( void ) {
  power_usart0_enable();
  // Enable receiver and transmitter
  UCSR0B |= ((1 << RXEN0) | (1 << TXEN0));
}

//-----------------------------------------------------------------------------
void USART_UnInit ( void ) {
  UCSR0B &= ~((1 << RXEN0) | (1 << TXEN0));
  power_usart0_disable();
  USART_DDR &= ~((1 << TXD_PINNR) | (1 << RXD_PINNR));
  USART_PORT &= ~((1 << TXD_PINNR) | (1 << RXD_PINNR));
}

//-----------------------------------------------------------------------------
void uart_putc ( char c ) {
  USART_Transmit( (uint8_t) c );
}

//-----------------------------------------------------------------------------
void uart_puts ( char *str ) {
  while (*str) {
    uart_putc(*str);
    str++;
  }
}

//-----------------------------------------------------------------------------
void uart_putline ( char *str ) {
  uart_puts(str);
  uart_putc('\r');
}

//-----------------------------------------------------------------------------
char uart_getc ( void ) {
  return (char) USART_Receive();
}

//-----------------------------------------------------------------------------
BOOL uart_isData ( void ) {
  return USART_IsDataAvailable() ? TRUE : FALSE;
}

//-----------------------------------------------------------------------------
void USART_Transmit ( uint8_t data ) {
  // Wait for empty transmit buffer
  while ( !( UCSR0A & (1<<UDRE0)) )
  ;
  // Put data into buffer, sends the data
  UDR0 = data;
}

//-----------------------------------------------------------------------------
uint8_t USART_Receive ( void ) {
  // Wait for data to be received
  while ( !(UCSR0A & (1<<RXC0)) )
  ;
  // Get and return received data from buffer
  return UDR0;
}

//-----------------------------------------------------------------------------
uint8_t USART_IsDataAvailable ( void ) {
  return UCSR0A & (1<<RXC0);
}

//-----------------------------------------------------------------------------
/*void USART_Flush ( void ) {
  uint8_t dummy;
  while ( UCSR0A & (1<<RXC0) ) dummy = UDR0;
}*/
