//-----------------------------------------------------------------------------
/// \file
/// fifo.c
/// \date
/// Create: 24.02.2009 Kae <br>
/// Modify: 02.02.2010 Kae
/// \brief
/// FIFO Puffer zur zwischenspeicherung der Busdaten
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// INCLUDES
//-----------------------------------------------------------------------------
#include "fifo.h"

//-----------------------------------------------------------------------------
// LOCAL VARIABLES
//-----------------------------------------------------------------------------
FIFO_BUFFER sReceiveFifo;


//-----------------------------------------------------------------------------
// LOCAL FUNCTIONS
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
/// \brief
/// Fuegt ein Datenbyte zum FIFO hinzu
/// \param
/// DataToAdd   Datenbyte
/// \param
/// *pFIFO      Pointer auf FIFO Struct
/// \return
/// FALSE wenn FIFO voll, ansonsten TRUE
//-----------------------------------------------------------------------------
BOOL fifo_AddData ( uint8_t dataToAdd, FIFO_BUFFER *pFIFO )
{ 
  if (((pFIFO->write) + 1 == pFIFO->read) || 
     ((pFIFO->read == 0) && ((pFIFO->write) + 1 == BUFFER_SIZE)))
  {
    return FALSE;
  }

  pFIFO->array[pFIFO->write] = dataToAdd;

  pFIFO->write++;
  if (pFIFO->write >= BUFFER_SIZE) {
    pFIFO->write = 0;
  }
  return TRUE;
}




//-----------------------------------------------------------------------------
/// \brief
/// Holt ein Byte vom FIFO und gibt diesen Speicher frei
/// \param
/// *pData    Pointer auf Datenvariabel
/// \param
/// *pFIFO    Pointer auf FIFO Struct
/// \return
/// FALSE wenn FIFO leer, ansonsten TRUE
//-----------------------------------------------------------------------------
BOOL fifo_GetData ( uint8_t *pData, FIFO_BUFFER *pFIFO )
{
  if (pFIFO->read == pFIFO->write) {
      return FALSE;
  }
  
  *pData = pFIFO->array[pFIFO->read];
  
  pFIFO->read++;
  if (pFIFO->read >= BUFFER_SIZE) {
      pFIFO->read = 0;
  }
  return TRUE;
}


//-----------------------------------------------------------------------------
/// \brief
/// Liest ein Byte vom FIFO, mit der Adresse relativ zur Leseadresse. Der Specher
/// wird nicht freigegeben und die Adressen nicht veraendert
/// \param
/// Addr      Adresse relativ zur Leseadresse (0 = aktuelle Leseadresse)
/// \param
/// *pFIFO    Pointer auf FIFO Struct
/// \return
/// Wert der an dieser Speicherstelle steht
//-----------------------------------------------------------------------------
#ifndef DO_NOT_USE_fifo_GetData
uint8_t fifo_ReadData ( uint8_t Addr, FIFO_BUFFER *pFIFO )
{
  Addr += pFIFO->read;
  
  if (Addr > BUFFER_SIZE) {
    Addr -= BUFFER_SIZE;
  }
  
  return pFIFO->array[Addr];
}
#endif


//-----------------------------------------------------------------------------
/// \brief
/// Liest die Anzahl verfuegbaren Daten im FIFO aus
/// \param
/// *pFIFO    Pointer auf FIFO Struct
/// \return
/// Anzahl Werte
//-----------------------------------------------------------------------------
uint8_t fifo_GetAvaiableDataCount ( FIFO_BUFFER *pFIFO )
{
  if (pFIFO->write == pFIFO->read) {
    return 0;
  } else {
    if (pFIFO->write > pFIFO->read) {
      return (pFIFO->write - pFIFO->read);
    } else {
      return (((pFIFO->write) + BUFFER_SIZE) - pFIFO->read);
    }
  }
}

//-----------------------------------------------------------------------------
// end of file
//-----------------------------------------------------------------------------
