/********
eeprom.c
*********/

//***********
//* Includes
//***********
#include "eeprom.h"

enum {
  eeAdr_active  = 0, // 0-1   uint16_t
  eeAdr_hour    = 2, // 2     uint8_t
  eeAdr_min     = 3, // 3     uint8_t
  eeAdr_days    = 4, // 4-10  BOOL
} EeAdresses;


/**
* \brief
* Liest die Daten aus dem EEPROM und speichert sie ins SRAM
*/
void EEPROM_To_SRAM ( Settings* data )
{
  unsigned char i;
  
  // Interrupts deaktivieren
  cli();
  
  data->active = EEPROM_read(eeAdr_active+1, 0x00, 0xff, 0x00) << 8;
  data->active += EEPROM_read(eeAdr_active, 0x00, 0xff, 0x00);
  if (data->active > 999) {
    data->active = 0;
  }
  
  data->hour = EEPROM_read(eeAdr_hour, 0, 23, 0);
  data->min = EEPROM_read(eeAdr_min, 0, 59, 0);
  
  for (i=0; i<7; i++) {
    data->days[i] = EEPROM_read(eeAdr_days+i, FALSE, TRUE, FALSE);
  }
  
  // Interrupts aktivieren
  sei();
}


/**
* \brief
* Speichert die Daten vom SRAM ins EEPROM
* \note
* Little-Endian format --> Lowest Byte first
*/
void SRAM_To_EEPROM ( Settings* data )
{
  unsigned char i;

  // Interrupts deaktivieren
  cli();
  
  EEPROM_write(eeAdr_active,    (uint8_t)(data->active & 0x00ff));
  EEPROM_write(eeAdr_active+1,  (uint8_t)((data->active & 0xff00)>>8));
  EEPROM_write(eeAdr_hour,      data->hour);
  EEPROM_write(eeAdr_min,       data->min);
  for (i=0; i<7; i++) {
    EEPROM_write(eeAdr_days+i,  data->days[i]);
  }
  
  // Interrupts aktivieren
  sei();
}


/**
* \brief
* Liest ein Byte aus dem internen EEPROM
* \param
* uiAddress    Adresse im EEPROM 0 - 511
* \param
* ucLowerBound  Minimalwert
* \param
* ucUpperBound  Maximalwert
* \param
* ucDefaultVal  Default Value falls Wert nicht innerhalb Minimal und Maximalwert
* \return
* Byte das ausgelesen wurde
*/
unsigned char EEPROM_read ( unsigned int uiAddress , 
              unsigned char ucLowerBound,
              unsigned char ucUpperBound,
              unsigned char ucDefaultVal )
{
  unsigned char Value;
  
  /* Wait for completion of previous write */
  while(EECR & (1<<EEPE)) {;}

  /* Set up address register */
  EEAR = uiAddress;

  /* Start eeprom read by writing EERE */
  EECR |= (1<<EERE);

  /* Return data from data register */
  Value = EEDR;
  
  /* Check if Value is Allowed */
  if ((Value >= ucLowerBound) && (Value <= ucUpperBound)) {
    return Value;
  } else {
    return ucDefaultVal;
  }
}


/**
* \brief
* Schreibt ein Byte in das internen EEPROM
* \param uiAddress Adresse im EEPROM 0 - 511
* \param ucData DAtenbyte
*/
void EEPROM_write ( unsigned int uiAddress, unsigned char ucData )
{
  /* Wait for completion of previous write */
  while(EECR & (1<<EEPE)) {;}

  /* Set up address and data registers */
  EEAR = uiAddress;
  EEDR = ucData;

  /* Write logical one to EEMWE */
  EECR |= (1<<EEMPE);

  /* Start eeprom write by setting EEWE */
  EECR |= (1<<EEPE);
}
