//-----------------------------------------------------------------------------
// INCLUDES
//-----------------------------------------------------------------------------
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <avr/pgmspace.h>

#include "application.h"
#include "eeprom.h"
#include "pump.h"
#include "digitalIo.h"
#include "led.h"
#include "uart.h"


//-----------------------------------------------------------------------------
// LOCAL FUNCTION DEKLARATIONS
//-----------------------------------------------------------------------------
static void   wrongValue  ( void );
static void   printTime   ( char *buf, uint8_t hour, uint8_t min );
static void   printNum    ( char *buf, uint16_t num );
static void   minTick     ( void );
static void   hourTick    ( void );


//-----------------------------------------------------------------------------
// VARIABELS
//-----------------------------------------------------------------------------
TimeAndDate   clock     = {0};
Settings      settings  = {0};

char tblWeekday[7][3] PROGMEM = {"So\0","Mo\0","Di\0","Mi\0","Do\0","Fr\0","Sa\0"};

char line1[] PROGMEM  = "Zeitschaltuhr\0";
char line2[] PROGMEM  = "------------------------------\0";
                       // yyyy.mm.dd - wd - hh:mm
char line21[] PROGMEM = "Akkuspannung:  \0"; // xxxx mV
char line3[] PROGMEM  = "Wochentage:    \0"; // xx  Tage
char line4[] PROGMEM  = "Einschaltzeit: \0"; // xxx Sekunden
char line5[] PROGMEM  = "Tageszeit:     \0"; // hh:mm
char line6[] PROGMEM  = "------------------------------\0";
char line7[] PROGMEM  = "   status\0";
char line8[] PROGMEM  = "   time hh:mm\0";
char line9[] PROGMEM  = "   weekday x (1=Mo 7=So)\0";
char line10[] PROGMEM = "   days xxxxxxx (Mo - So) 0/1\0";
char line11[] PROGMEM = "   active sss\0";
char line12[] PROGMEM = "   daytime hh:mm\0";

PGM_P txtTable[] PROGMEM = {
  line1, line2, line6, line7,
  line8, line9, line10, line11,
  line12
};


//-----------------------------------------------------------------------------
void initApp ( void ) {
  voltageControlTick(&settings);
  EEPROM_To_SRAM(&settings);
}

//-----------------------------------------------------------------------------
void setTime ( uint8_t hour, uint8_t min ) {
  if ((hour <= 60) && (min <= 60)) {
    clock.sec = 0;
    clock.hour = hour;
    clock.min = min;
  } else {
    wrongValue();
  }
}

//-----------------------------------------------------------------------------
void setWeekday ( uint8_t wDay ) {
  // wDay           1 Mo 7 So
  // clock.weekday  0 So 6 Sa
  if ((wDay >= 1) && (wDay <= 7)) {
    if (wDay == 7) {
      wDay = 0;
    }
    clock.weekday = wDay;
  } else {
    wrongValue();
  }
}

//-----------------------------------------------------------------------------
void setDays ( BOOL arr[] ) {
  memcpy(settings.days, arr, 7);
}

//-----------------------------------------------------------------------------
void setActiveTime ( uint16_t time ) {
  settings.active = time;
}

//-----------------------------------------------------------------------------
void setDayTime ( uint8_t hour, uint8_t min ) {
  if ((hour <= 60) && (min <= 60)) {
    settings.hour = hour;
    settings.min = min;
  } else {
    wrongValue();
  }
}

//-----------------------------------------------------------------------------
void sendStatus ( void ) {
  uint8_t   i;
  uint8_t   cnt;
  char      line[50];
  char      weekday[3];

  // Kopf
  for (i=0; i<2; i++) {
    strcpy_P(line, (const char*) pgm_read_word(&txtTable[i]));
    uart_putline(line);
  }
  
  // Wochentag und Zeit
  memcpy_P(weekday, &tblWeekday[clock.weekday][0], 3);
  line[0] = 0;
  printTime(line, clock.hour, clock.min);
  strcat(line, " - ");
  strcat(line, weekday);
  uart_putline(line);
  
  // Spannung
  strcpy_P(line, line21);
  printNum(line, settings.voltage);
  strcat(line, " mV");
  uart_putline(line);

  // Wochentage an denen Eingeschaltet
  strcpy_P(line, line3);
  cnt = 0;
  for (i=1; i<8; i++) {
    if (settings.days[i%7] == TRUE) {
      memcpy_P(weekday, &tblWeekday[i%7][0], 3);
      cnt++;
      if (cnt != 1) {
        strcat(line, " - ");
      }
      strcat(line, weekday);
    }
  }
  if (cnt == 0) {
    strcat(line, "Keine");
  }
  uart_putline(line);

  // Einschaltzeit
  strcpy_P(line, line4);
  printNum(line, settings.active);
  strcat(line, " Sekunden");
  uart_putline(line);

  // Tageszeit
  strcpy_P(line, line5);
  printTime(line, settings.hour, settings.min);
  uart_putline(line);
  
  // Befehle
  for (i=2; i<9; i++) {
    strcpy_P(line, (const char*) pgm_read_word(&txtTable[i]));
    uart_putline(line);
  }

  SRAM_To_EEPROM(&settings);
}

//-----------------------------------------------------------------------------
void secTick ( void ) {
  clock.sec++;
  if (clock.sec == 60) {
    minTick();
  }
  
  setPump(getKey());
  
  checkVoltageAvaiableTick();
  pumpControlSecTick(&clock, &settings);
  ledControlTick(&settings);
  digitalIoSecTick();
}

//-----------------------------------------------------------------------------
void checkVoltageAvaiableTick ( void ) {
  /* Funktioniert in dieser Hardwarekonfiguration nicht
  getVoltageAvailable(); */
}


//-----------------------------------------------------------------------------
// LOCAL FUNCTIONS
//-----------------------------------------------------------------------------

static void minTick ( void ) {
  clock.sec = 0;
  clock.min++;
  
  if (clock.min == 60) {
    hourTick();
  }
  
  voltageControlTick(&settings);
}

//-----------------------------------------------------------------------------
static void hourTick ( void ) {
  clock.min = 0;
  clock.hour++;

  if (clock.hour == 24) {
    clock.hour = 0;
    
    // Wochentag hochzhlen
    clock.weekday++;
    if (clock.weekday > 6) {
      clock.weekday = 0;
    }
  }
}

//-----------------------------------------------------------------------------
static void wrongValue ( void ) {
  uart_putline("Falscher Wertebereich !");
}

//-----------------------------------------------------------------------------
/// \brief
/// Haengt die Zeit an den String im Format hh:mm an. Wenn noetig mit fuehrender 0
static void printTime ( char *buf, uint8_t hour, uint8_t min ) {
  char num[3];
  
  itoa(hour, num, 10);
  if (hour < 10) {
    strcat(buf, "0");
  }
  strcat(buf, num);
  strcat(buf, ":");
  
  itoa(min, num, 10);
  if (min < 10) {
    strcat(buf, "0");
  }
  strcat(buf, num);
}

//-----------------------------------------------------------------------------
/// \brief
/// Haengt die eine Zahl an den String an.
static void printNum ( char *buf, uint16_t number ) {
  char str[10];
  
  itoa(number, str, 10);
  strcat(buf, str);
}
