//-----------------------------------------------------------------------------
// INCLUDES
//-----------------------------------------------------------------------------
#include <string.h>
#include <stdlib.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>

#include "cmd_parser.h"
#include "stdDefs.h"
#include "fifo.h"
#include "uart.h"
#include "application.h"


//-----------------------------------------------------------------------------
// DEFINES
//-----------------------------------------------------------------------------
#define AMOUNT_OF_CMDS  6     // Anzahl Befehle
#define MAX_CMD_LEN     50    // Maximale Lnge eines Befehls + Sicherheit
#define DC              0x01  // entspricht Don't Care

#define code
typedef void (code *pFunction) ( char str[], uint8_t endPos );


//-----------------------------------------------------------------------------
// LOCAL FUNCTION DEKLARATIONS
//-----------------------------------------------------------------------------
static void   runCmdTime        ( char str[], uint8_t endPos );
static void   runCmdWeekday     ( char str[], uint8_t endPos );
static void   runCmdDays        ( char str[], uint8_t endPos );
static void   runCmdPeriod      ( char str[], uint8_t endPos );
static void   runCmdDayTime     ( char str[], uint8_t endPos );
static void   runCmdSendStatus  ( char str[], uint8_t endPos );
static void   cmdUnknown        ( void );


//-----------------------------------------------------------------------------
// ENUMERATIONS AND TYPES
//-----------------------------------------------------------------------------
enum {
  Status,
  DayTime,
  Time,
  Date,
  Days,
  Period,
} Command;


//-----------------------------------------------------------------------------
// VARIABELS
//-----------------------------------------------------------------------------
                            /// status
char cmdStatus[]  PROGMEM = {'s', 't', 'a', 't', 'u', 's', '\0'};
                            /// daytime hh:mm
char cmdDayTime[] PROGMEM = {'d', 'a', 'y', 't', 'i', 'm', 'e', ' ', DC, DC, ':', DC, DC, '\0'};
                            /// time hh:mm
char cmdTime[]    PROGMEM = {'t', 'i', 'm', 'e', ' ', DC, DC, ':', DC, DC, '\0'}; 
                            /// weekday x
char cmdWeekday[] PROGMEM = {'w', 'e', 'e', 'k', 'd', 'a', 'y', ' ', DC, '\0'};
                            /// days xxxxxxx
char cmdDays[]    PROGMEM = {'d', 'a', 'y', 's', ' ', DC, DC, DC, DC, DC, DC, DC, '\0'};
                            /// period sss
char cmdPeriod[]  PROGMEM = {'a', 'c', 't', 'i', 'v', 'e', ' ', DC, DC, DC, '\0'};


/// Table with Pointers to the Command Strings
PGM_P cmdPointerTable[AMOUNT_OF_CMDS] PROGMEM = {
  cmdStatus,
  cmdDayTime,
  cmdTime,
  cmdWeekday,
  cmdDays,
  cmdPeriod,
};

/// Reihenfolge muss mit Command enum uebereinstimmen
pFunction fctPointerTable[AMOUNT_OF_CMDS] PROGMEM = {
  runCmdSendStatus,
  runCmdDayTime,
  runCmdTime,
  runCmdWeekday,
  runCmdDays,
  runCmdPeriod,
};

/// Dieses Flag wird geetzt sobald ein '\r' Byte empfangen wurde
BOOL  endFlag = FALSE;


//-----------------------------------------------------------------------------
void setEndFlag ( void ) {
  endFlag = TRUE;
}

//-----------------------------------------------------------------------------
BOOL getEndFlag ( void ) {
  return endFlag;
}

//-----------------------------------------------------------------------------
void copyAndInterpretCommand ( void ) {
  BOOL      found;
  uint8_t   i, n;
  uint8_t   cmpLen;
  uint8_t   inEndPos;
  uint8_t   cmdNum = 0xff;
  char      inBuffer[MAX_CMD_LEN];
  char      cmpBuffer[MAX_CMD_LEN];
  pFunction fctPntr;     
  
  // Befehl aus RxD Buffer in lokalen kopieren (Es darf kein Interrupt auftreten)
  n = Rx_GetAvaiableDataCount();
  for(i=0; i<n; i++) {
    Rx_GetData(&inBuffer[i]);
  }
  
  // Endposition des Befehls ermitteln
  inEndPos = 0;
  while (inBuffer[inEndPos] != '\r') {
    inEndPos++;
  }
  
  for (i=0; i<AMOUNT_OF_CMDS; i++) {
    // Befehle aus FLASH in RAM Buffer kopieren
    strcpy_P(cmpBuffer, (const char*) pgm_read_word(&cmdPointerTable[i]));
    
    // Befehle vergleichen
    found = TRUE;
    cmpLen = strlen(cmpBuffer);
    for (n=1; n<cmpLen; n++) {
      if (!((inBuffer[inEndPos - n] == cmpBuffer[cmpLen - n]) 
          || (cmpBuffer[cmpLen - n] == DC)))
      {
        found = FALSE;
        break;
      }
    }
    
    // Befehl und '>' mssen bereinstimmen, also keine berflssigen Zeichen vor dem Befehl
    if ((found == TRUE) && (inBuffer[inEndPos - cmpLen - 1] == '>')) {
      cmdNum = i;
      break;
    }
  }
  
  // Befehl auswerten
  if (cmdNum < AMOUNT_OF_CMDS) {
    fctPntr = (pFunction)pgm_read_word(&fctPointerTable[cmdNum]);
    fctPntr(inBuffer, inEndPos);
  } else {
    cmdUnknown();
  }
}


//-----------------------------------------------------------------------------
// LOCAL FUNCTIONS
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
/// \note
/// Gilt fr alle Funktionen dieser Art
/// \brief
/// Liest die Zeit aus dem Befehl aus und ruft die Funktion auf um diese einzustellen
/// \param
/// str[]     Command String
/// \param
/// endPos    Position des letzten Zeichens ('\r') im Command String
static void runCmdTime ( char str[], uint8_t endPos ) {
  char      sHour[3] = {0};
  char      sMin[3]  = {0};
  
  memcpy(sHour, str + endPos - 5, 2);
  memcpy(sMin, str + endPos - 2, 2);
  setTime(  (uint8_t) atoi(sHour), 
            (uint8_t) atoi(sMin));
}

//-----------------------------------------------------------------------------
static void runCmdWeekday ( char str[], uint8_t endPos ) {
  char      sDay[2]   = {0};
  uint8_t   wDay;
  
  sDay[0] = *(str + endPos - 1);
  wDay = (uint8_t) atoi(sDay);
  setWeekday(wDay);
}

//-----------------------------------------------------------------------------
static void runCmdDays ( char str[], uint8_t endPos ) {
  char      sDays[7];   // Mo - So
  BOOL      bDays[7];   // So - Sa
  uint8_t   i;
  uint8_t   j = 1;
  
  memcpy(sDays, str + endPos - 7, 7);
  for (i=0; i<7; i++) {
    if (sDays[i] == '1') {
      bDays[j] = TRUE;
    } else if (sDays[i] == '0') {
      bDays[j] = FALSE;
    } else {
      cmdUnknown();
      return;
    }
    
    j++;
    if (j > 6) {
      j = 0;
    }
  }
  
  setDays(bDays);
}

//-----------------------------------------------------------------------------
static void runCmdPeriod ( char str[], uint8_t endPos ) {
  char      sPeriod[4] = {0};
  
  memcpy(sPeriod, str + endPos - 3, 3);
  setActiveTime( atoi(sPeriod) );
}

//-----------------------------------------------------------------------------
static void runCmdDayTime ( char str[], uint8_t endPos ) {
  char      sHour[3] = {0};
  char      sMin[3]  = {0};
  
  memcpy(sHour, str + endPos - 5, 2);
  memcpy(sMin, str + endPos - 2, 2);
  setDayTime( (uint8_t) atoi(sHour), 
              (uint8_t) atoi(sMin));
}

//-----------------------------------------------------------------------------
static void runCmdSendStatus ( char str[], uint8_t endPos ) {
  sendStatus();
}

//-----------------------------------------------------------------------------
static void cmdUnknown ( void ) {
  uart_putline("Unbekannter Befehl !");
}
