/*******************************************************

					USB.c

********************************************************/

#include "USB.h"

volatile uint8_t usbRxByte=0;			// Empfangenes USB Byte
volatile uint8_t usbTxByte=0;			// Zu schreibendes USB Byte
volatile uint8_t usbRxError=0;			// USB Receive Error Flag
volatile uint8_t usbTxError=0;			// USB Transmit Error Flag
volatile uint8_t usbMsgState=ST_START;	// Schrittmerker fr den Empfang der USB Daten
volatile uint16_t usbMsgLength=0;		// Lnge des Datenbuffers
volatile uint8_t usb_initialisiert=0;	// Flag fr FT245 (ntig fr: siehe usbInit())
volatile uint8_t usb_timer0_start=0;		// Flag fr Start von Timer0
volatile uint8_t usb_timer0_ovf=0;		// Counter fr Overflows von Timer0


cBuffer USBRxBuffer;	// Receive Buffer
cBuffer USBTxBuffer;	// Transmit Buffer

uint8_t USBRxDATA[USB_RX_BUFFER_SIZE];	// Datenbuffer fr Receive
uint8_t USBTxDATA[USB_TX_BUFFER_SIZE];	// Datenbuffer fr Transmit

// Pulse nach z.B. Power On Reset an
// TXE und RXF des FT245 abwarten
// Genauere Beschreibung siehe usbInit()
void usb_ft245_pulses_init(void)
{
	// verschachtelte Schleifen
	// Dauer ca. 200ms
	for(uint16_t i=0;i<2000;i++)
		for(uint8_t j=0;j<255;j++)
			if(FT245_TXE_HIGH())	// Funktionsende bei erneutem 
				return;				// High-Pegel an TXE
									// bei nchstem TXE-Interrupt wird
									// Funktion erneut aufgerufen
	
	// TXE ist an dieser Stelle auch nach den 200ms immer noch low
	
	EIMSK |= (1<<INT4); 	// Jetzt auch Ext. Interrupt fr RXF aktivieren
	if(EIFR & (1<<INTF4))	// Interrupt Flag lschen falls gesetzt
		EIFR |= (1<<INTF4);

	usbMsgState = ST_START;	// Merker fr USB Message zurcksetzen
	usb_initialisiert = 1; 	// Flag setzen, damit diese Funktion nicht
							// bei jedem TXE Interrupt aufgerufen wird
}

void usbInit(void)
{		
	// RX und TX Buffer initalisieren
	bufferInit(&USBRxBuffer, USBRxDATA, USB_RX_BUFFER_SIZE);
	bufferInit(&USBTxBuffer, USBTxDATA, USB_TX_BUFFER_SIZE);
	
	// Nach Power On Reset oder nach erneutem Anstecken des
	// USB Interfaces des Webservers am PC mssen an den Pins
	// TXE bzw RXF vom FT245 einige Pulse abgewartet werden,  
	// bis der FT245 vollstndig am PC initialisiert ist.
	// Die Pulse drfen nicht als Daten erkannt werden, da
	// sonst das bertragungsprotokoll nicht mehr richtig
	// funktioniert.
	// Im Flag "usb_initialisiert" wird gespeichert, ob diese
	// Initialisierung bereits durchgefhrt wurde.
	// Um die Pulse abzuwarten, orientiert sich das Programm
	// am Pin TXE. Wenn dieser Pin fr ca. 200..250ms low ist,
	// dann wird angenommen, dass der FT245 initialisiert ist
	// und keine neuen Pulse, die nicht infolge von Daten
	// heinkommen, mehr kommen. Deshalb wird zu Beginn auch
	// nur der Externe Interrupt fr TXE (INT5) aktiviert.
	// Der Interrupt fr RXF (Int4) wird erst nach diesem
	// Init-Vorgang aktiviert.
	
	// Disable Interrupts
	cli();
	
	// FT245 initialisieren
	ft245_init();
	
	// Externe Interrupts 4 und 5 initialisieren
	// sense Falling Edge for Int4 (RXF)
	// Any logical change for Int5 (TXE)
	EICRB |= (1<<ISC41) | (1<<ISC50);
	EICRB &= ~((1<<ISC40) | (1<<ISC51));
	
	// External Interrupt 5 aktivieren (TXE)
	EIMSK |= (1<<INT5);
	
	if(usb_initialisiert) 			// FT245-Pulse bereits abgewartet,
		EIMSK |= (1<<INT4);		// dann auch INT4 fr RXF aktivieren
	else if(FT245_TXE_LOW())		// sonst Init Funktion zum Abwarten dieser
		usb_ft245_pulses_init();	// Pulse aufrufen, Wenn TXE bereits low.
									// Sonst geschieht dies durch den 1. TXE-Interrupt 
	// Enable Interrupts
	sei();
}

uint8_t usbGetByte(uint8_t* byte)
{
	if(USBRxBuffer.size)
	{
		if(USBRxBuffer.datalength)
		{
			*byte = bufferGetFromFront(&USBRxBuffer);
			return 1;
		}
		else
			return 0;
	}
	else
		return 0;
}

uint8_t usbWriteByte(uint8_t byte)
{
	if(bufferAddToEnd(&USBTxBuffer,byte))
	{
		if(FT245_TXE_LOW())
		{
			cli();
			EIFR |= (1<<INTF5);
			sei();
		}
		return 1;
	}
	
	else
		return 0;
}

// Funkion: berprfe ob neue DATEN vorhanden
void usbNewData(void)
{
	cli();
	
	if((USBTxBuffer.datalength && FT245_TXE_LOW()) || usbTxError)
		EIFR |= (1<<INTF5);
	
	if((USBRxBuffer.datalength<USBRxBuffer.size && FT245_RXF_LOW()) || usbRxError)
		EIFR |= (1<<INTF4);
	
	sei();
}

// Receive
ISR(INT4_vect)
{		
	if(usbRxError)
	{
		if(bufferAddToEnd(&USBRxBuffer,usbRxByte))
			usbRxError=0;
		else
			return;
	}
	
	while(FT245_RXF_LOW())
	{

		if(!(ft245_get_byte(&usbRxByte)))	//Auslesen Fehlgeschlagen
			break;
		
		switch(usbMsgState)
		{
		case ST_START:
#if COMM_MODE == USB_FDXX			
			if(usbRxByte==MESSAGE_START)
				collectCompleteMessage();
#endif
			if(!bufferAddToEnd(&USBRxBuffer,usbRxByte))
				usbRxError=1;					
			usbMsgState=ST_GET_SEQ_NUM;
			break;
			
		case ST_GET_SEQ_NUM:	
			if(!bufferAddToEnd(&USBRxBuffer,usbRxByte))
				usbRxError=1;					
			usbMsgState=ST_MSG_SIZE_1;
			break;		
			
		case ST_MSG_SIZE_1:
			usbMsgLength = usbRxByte<<8;
			if(!bufferAddToEnd(&USBRxBuffer,usbRxByte))
				usbRxError=1;					
			usbMsgState=ST_MSG_SIZE_2;
			break;
			
		case ST_MSG_SIZE_2:
			usbMsgLength |= usbRxByte;
			if(!bufferAddToEnd(&USBRxBuffer,usbRxByte))
				usbRxError=1;
			usbMsgState = ST_GET_DATA;
			break;
			
		case ST_GET_DATA:
			if(!bufferAddToEnd(&USBRxBuffer,usbRxByte))
				usbRxError=1;
			if(usbMsgLength-- == 0)
				usbMsgState=ST_GET_CHECK;
			break;
			
		case ST_GET_CHECK:
			if(!bufferAddToEnd(&USBRxBuffer,usbRxByte))
				usbRxError=1;
			usbMsgState = ST_START;
			break;
		}
	} // while
}

// Timer0 Overflow
ISR(TIMER0_OVF_vect)
{
	if(++usb_timer0_ovf == 16) // nach ca. 1s
	{
		EIMSK &= ~(1<<INT4); 	// Ext. Interrupt fr RXF deaktivieren
		if(EIFR & (1<<INTF4))	// Interrupt Flag lschen falls gesetzt
			EIFR |= (1<<INTF4);
		
		usbMsgState = ST_START;	// Merker fr USB Message zurcksetzen
		
		usb_initialisiert = 0;
		usb_timer0_ovf = 0;
		TCCR0 = 0x00;	// Timer stopp
		TIMSK &= ~(1<<TOIE0);	// Overflow interrupt deaktivieren
	}
}

// Transmit
ISR(INT5_vect)
{
	if(FT245_TXE_HIGH() && usb_initialisiert)	// Rising edge on TXE
	{
		TCCR0 |= (1<<CS00)|(1<<CS01)|(1<<CS02);	// Starte Timer0 mit fosc/1024
		TIMSK |= (1<<TOIE0);	// Overflow interrupt aktivieren
		usb_timer0_start = 1;
		usb_timer0_ovf = 0;
		return;
	}
	else if(usb_timer0_start==1) // bei einer Falling edge
	{
		TCCR0 = 0x00;	// Timer stopp
		TIMSK &= ~(1<<TOIE0);	// Overflow interrupt deaktivieren
		usb_timer0_start = 0;
		usb_timer0_ovf = 0;
	}
	
	// Nach Power On Reset oder nach erneutem Anstecken des
	// USB Interfaces des Webservers am PC mssen an den Pins
	// TXE bzw RXF vom FT245 einige Pulse abgewartet werden, 
	// bis der FT245 vollstndig am PC initialisiert ist.
	// Die Pulse drfen nicht als Daten erkannt werden, da
	// sonst das bertragungsprotokoll nicht mehr funktioniert.
	
	if(!usb_initialisiert) 			// Flag, in dem gespeichert wird, ob die
	{								// FT245-Pulse bereits abgewartet wurden.
		usb_ft245_pulses_init();	// Falls nicht, ntige Funktion aufrufen
		return;						// Interruptroutine verlassen
	}
		
	// sei() + Interrupt (EXTINT5) deaktivieren?
	if(usbTxError)
	{
		if(ft245_write_byte(usbTxByte))
			usbTxError = 0;
		else
			return;
	}
	
	while(FT245_TXE_LOW())
	{
		if(USBTxBuffer.datalength)	// Noch Daten im Buffer vorhanden
		{
			usbTxByte = bufferGetFromFront(&USBTxBuffer);
			if(!ft245_write_byte(usbTxByte)) // Schreiben fehlgeschlagen
			{
				usbTxError = 1;
				break;
			}
			else
				usbTxError = 0;
		}
		else
			break;
	}
}

