/****************************************************************************
	COMMUNICATION.H

	Defines for Communication with AVR Studio

	Only important for user:
		COMM_MODE:	Defines if either RS232 oder USB is used

		If USB is used: 
			FT245_ADRESS:	Selects the Adress of the FT245 in the external
							memory interface of ATMEGA128
*****************************************************************************/
#ifndef _COMMUNICATION_H
#define _COMMUNICATION_H

#include <avr/io.h>

#define USB_FDXX	0x00
#define	RS232		0x01

#define COMM_MODE	USB_FDXX

 /* UART Baudrate, AVRStudio AVRISP only accepts 115200 bps */

#if COMM_MODE == RS232

#define BAUDRATE 115200

 /*  Enable (1) or disable (0) USART double speed operation */

#define UART_BAUDRATE_DOUBLE_SPEED 0

 /*  Defines for the various USART registers */

#if  defined(__AVR_ATmega8__)    || defined(__AVR_ATmega16__) || defined(__AVR_ATmega32__) \
  || defined(__AVR_ATmega8515__) || defined(__AVR_ATmega8535__) \
/* ATMega8 with one USART */
#define	UART_BAUD_RATE_LOW	     UBRRL
#define	UART_STATUS_REG	         UCSRA
#define	UART_CONTROL_REG	     UCSRB
#define	UART_ENABLE_TRANSMITTER	 TXEN
#define	UART_ENABLE_RECEIVER	 RXEN
#define	UART_TRANSMIT_COMPLETE	 TXC
#define	UART_RECEIVE_COMPLETE	 RXC
#define	UART_DATA_REG	         UDR
#define UART_DOUBLE_SPEED        U2X

#elif  defined(__AVR_ATmega64__) || defined(__AVR_ATmega128__) || defined(__AVR_ATmega162__) || defined(__AVR_ATmega88__) || defined(__AVR_ATmega168__)
/* ATMega with two USART */
#define	UART_BAUD_RATE_LOW	     UBRR0L
#define	UART_STATUS_REG	         UCSR0A
#define	UART_CONTROL_REG	     UCSR0B
#define	UART_ENABLE_TRANSMITTER	 TXEN0
#define	UART_ENABLE_RECEIVER	 RXEN0
#define	UART_TRANSMIT_COMPLETE	 TXC0
#define	UART_RECEIVE_COMPLETE	 RXC0
#define	UART_DATA_REG	         UDR0
#define UART_DOUBLE_SPEED        U2X0

#else
	#error "no UART definition for MCU available"
#endif

 /* Macro to calculate UBBR from XTAL and baudrate */

#if UART_BAUDRATE_DOUBLE_SPEED
#define UART_BAUD_SELECT(baudRate,xtalCpu) (((float)(xtalCpu))/(((float)(baudRate))*8.0)-1.0+0.5)
#else
#define UART_BAUD_SELECT(baudRate,xtalCpu) (((float)(xtalCpu))/(((float)(baudRate))*16.0)-1.0+0.5)
#endif

#endif


#if COMM_MODE == USB_FDXX

// Adresse des FT245 am Adressbus
// kann zwischen 0xA000 und 0xAFFF liegen
#define FT245_ADRESS 0xA000

// Definition fr TXE
#define TXE_DDR		DDRE
#define TXE_PORT	PORTE
#define TXE_PIN		PINE
#define TXE_BIT		5

// Definition fr RXF
#define RXF_DDR		DDRE
#define RXF_PORT	PORTE
#define RXF_PIN		PINE
#define RXF_BIT		4

#define	ft245_RXF_LOW()	 (!(RXF_PIN & (1<<RXF_BIT)))
#define	ft245_RXF_HIGH() (RXF_PIN & (1<<RXF_BIT))
#define ft245_TXE_LOW()	 (!(TXE_PIN & (1<<TXE_BIT)))
#define ft245_TXE_HIGH() (TXE_PIN & (1<<TXE_BIT))

#endif

/********************************************************
*														*
*			Funktionsprototypen							*
*														*
*********************************************************/

/********************************************************
	Schreibt ein Byte ber RS232 oder in den FIFO des 
	FT245

Parameter:		c		Datenbyte, das bertragen
						werden soll

Rckgabewert:	NONE
*********************************************************/

void sendchar(char c);

/********************************************************
	Liest ein Byte ber RS232 aus oder aus dem FIFO des 
	FT245

Parameter:		NONE

Rckgabewert:	unsigned char	ausgelesenes Datenbyte
*********************************************************/

unsigned char recchar(void);

/********************************************************
	berprft ob neue Daten ber RS232 im  FIFO des FT245
	vorhanden sind

Parameter:		NONE

Rckgabewert:	unsigned char  TRUE: neue Daten vorhanden
							   FALSE: keine neuen Daten
*********************************************************/

unsigned char newDataAvailable(void);

#endif
